<?php

namespace CocktailRecipes\Core\Base;

if (!defined('ABSPATH')) exit;

use CocktailRecipes\Plugin;
use CocktailRecipes\Core\Helpers\HTML;
use CocktailRecipes\Core\Helpers\Logger;

abstract class Shortcode
{
    // ---------------------------------------------------------------------
    // Shortcode settings
    // ---------------------------------------------------------------------
    // These settings can be overridden in a shortcode subclass to adjust
    // behaviors. These are only supported for instance invocation, not
    // with statically invoked shortcodes.

    /** True to suppress WordPress preprocessing of shortcode content */
    public const RAW_CONTENT = false;


    // ---------------------------------------------------------------------
    // Class properties
    // ---------------------------------------------------------------------

    // Call parameters for use during instance invocation (read-only)
    protected string $tag;
    protected array $atts;
    protected ?string $content;

    // Saved one-time output for current request (internal use only)
    private static array $saved = [];


    // ---------------------------------------------------------------------
    // Methods to override in subclasses
    // ---------------------------------------------------------------------
    // When the default render() function is used for static invocation, it
    // must be extended in the child class to return the shortcode output.
    // If an alternate static method is used it must be added with the same
    // arguments as render(). For instance invocation, the output() method
    // must be overwritten to provide the output of the shortcode.

    /** Default output generation function for static invocation */
    public static function render($atts, $content = null, $tag = ''): string
    {
        return '';
    }

    /** Output generation function when handled as an invokable object */
    protected function output(): string
    {
        return '';
    }


    // ---------------------------------------------------------------------
    // Optional methods to override in subclasses
    // ---------------------------------------------------------------------
    // For instance invocation, the following functions can be extended to
    // support optional features as needed by the shortcode.

    /**
     * Optional subclass initialization hook
     *
     * Called after shortcode attributes, content and tag are stored to the
     * object by the consturctor. Override in child class to perform custom
     * setup or normalization without replacing the constructor.
     */
    protected function init(): void {}

    /**
     * Define a unique rendering key to enable one-time output generation
     *
     * If a string or true is returned, output will be saved for the current
     * request so if the shortcode is used again on the page it will not be
     * built again. When true the default key for the current shortcode will
     * be used, otherwise a custom key can be returned.
     *
     * Default key is tag plus md5 of sorted attributes & content
     *
     * @return string|bool  cache key (string), default key (true), disabled (false)
     */
    protected function renderOnce()
    {
        return false;
    }

    // ---------------------------------------------------------------------
    // Helper functions
    // ---------------------------------------------------------------------

    /** Get saved shortcode output if it was stored with saveOutput() */
    final protected static function getSaved(string $key): ?string
    {
        return self::$saved[$key] ?? null;
    }

    /** Save and return shortcode output for reuse during request */
    final protected static function saveOutput(string $key, string $value): string
    {
        return self::$saved[$key] = $value;
    }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------
    // Supported SHORTCODES formats:
    //    tag => className                  calls <class>::render()
    //    tag => [className, methodName]    calls <class>::<method>()
    //    tag => ['new', className]         calls (new <class>())()

    /** Shortcode handler */
    final public static function handler($atts, $content = null, $tag = ''): string
    {
        $handler = Plugin::SHORTCODES[$tag] ?? null;
        if (!$handler) return HTML::comment("unknown shortcode [$tag]");
        if (is_string($handler)) {
            $class = $handler;
            $func  = 'render';
        } elseif (is_array($handler) && count($handler) >= 2) {
            [$class, $func] = $handler;
        } else {
            return HTML::comment("invalid handler for shortcode [$tag]");
        }
        if (!is_string($tag)) {
            Logger::warning(static::class . "received non-string tag (" . gettype($tag) . ")");
            $tag = '?';
        }
        if (!is_array($atts)) {
            Logger::warning("Shortcode [$tag] received non-array atts (" . gettype($atts) . ")");
            $atts = [];
        }
        if ($content !== null && !is_string($content)) {
            Logger::warning("Shortcode [$tag] received non-string content (" . gettype($content) . ")");
            $content = null;
        }
        // Note: We are intentionally not checking class_exists() or method_exists()
        // to avoid overhead on every shortcode render. Incorrect setup of any
        // shortcodes in the Plugin should surface during development.
        return ($class === 'new')
            ? (new $func($atts, $content, $tag))()
            : $class::$func($atts, $content, $tag);
    }

    /** Register filter to handle raw content as needed for specific plugins */
    final public static function registerRawContent(array $tags): void
    {
        if (!$tags) return;

        // regex pattern to match [tag]...[/tag]
        // $m[1] = tag with attributes
        // $m[2] = tag
        // $m[3] = content
        $pattern = '#\[((' . implode('|', $tags) . ')(?:\s+[^\]]*)?)\](.*?)\[/\2\]#s';

        add_filter('the_content', function ($content) use ($tags, $pattern) {
            if ($content === null || $content == '') return $content;

            // quick check for any "raw" shortcode tag
            $found = false;
            foreach ($tags as $tag) {
                if (strpos($content, "[$tag]") !== false || strpos($content, "[$tag ") !== false) {
                    $found = true;
                    break;
                }
            }
            if (!$found) return $content;

            // wrap contents in <pre>...</pre> to prevent WordPress text mangling
            return preg_replace_callback(
                $pattern,
                fn($m) => '[' . $m[1] . ']<pre data-raw-content>' . $m[3] . '</pre>[/' . $m[2] . ']',
                $content
            );
        }, 1);  // priority 1 ensures it runs before wptexturize(), etc.
    }

    // --- Instance invocation ---

    final protected function __construct(array $atts = [], ?string $content = null, string $tag = '')
    {
        $this->tag     = $tag;
        $this->atts    = $atts;
        $this->content = $content ?? '';
        if (static::RAW_CONTENT) {
            $this->content = preg_replace('#^\s*<pre data-raw-content>(.*)</pre>\s*$#s', '$1', $this->content);
        }
        $this->init();
    }

    final public function __invoke(): string
    {
        if (!($key = $this->renderOnce())) return $this->output();
        if ($key === true) {
            // use default key from shortcode tag, attributes and content
            ksort($this->atts);
            $key = $this->tag . md5(json_encode($this->atts) . $this->content);
        } elseif (!is_string($key)) {
            return $this->output();
        }
        return self::$saved[$key] ??= $this->output();
    }
}
