<?php

namespace CocktailRecipes\Core\Base;

if (!defined('ABSPATH')) exit;

require_once __DIR__ . '/ReadOnlyProps.php';
use CocktailRecipes\Core\Admin\Page as AdminPage;
use CocktailRecipes\Core\Admin\Settings;
use CocktailRecipes\Core\Base\Job as JobDispatcher;
use CocktailRecipes\Core\Base\Shortcode as ShortcodeHandler;
use CocktailRecipes\Core\Helpers\Assets;
use Plugin_Upgrader;
use LogicException;

/**
 * Abstract Plugin
 *
 * @property-read   string  $name           plugin name
 * @property-read   string  $path           absolute path to plugin root
 * @property-read   string  $slug           plugin slug
 * @property-read   string  $namespace      plugin namespace
 * @property-read   string  $text_domain    localization text domain
 * @property-read   string  $url            public url to plugin root
 * @property-read   string  $version        plugin version
 */
abstract class Plugin extends ReadOnlyProps
{
    // ---------------------------------------------------------------------
    // Class constants
    // ---------------------------------------------------------------------
    // Most of these constants should be overridden in the child class to
    // define how the plugin will be initialized, assets to be loaded, etc.

    /**
     * Frontend stylesheet file(s)
     *
     * One or more stylesheet assets in a mix of these formats:
     *    asset
     *    name => asset
     *    name => [asset, ...]
     *
     * Without a name the asset is a general asset for the plugin and is
     * enqueued on all frontend (i.e. non-admin) pages. If a name is used
     * the asset is registered with WordPress but not enqueued, which will
     * allow it to be conditionally included with Assets::include(<name>)
     * from the addFrontendAssets() and/or addAdminAssets() functions.
     *
     * Each asset can be listed as:
     *    name    asset/css/<name>.css
     *   ./file   full path to file, relative to plugin root
     *   /file    full path to file, relative to WordPress root
     *   url      full URL to file
     */
    protected const STYLESHEETS = [];

    /**
     * Frontend JavaScript file(s)
     *
     * One or more JavaScript assets in a mix of these formats:
     *    asset
     *    name => asset
     *    name => [asset, ...]
     *
     * Without a name the asset is a general asset for the plugin and is
     * enqueued on all frontend (i.e. non-admin) pages. If a name is used
     * the asset is registered with WordPress but not enqueued, which will
     * allow it to be conditionally included with Assets::include(<name>)
     * from the addFrontendAssets() and/or addAdminAssets() functions.
     *
     * Each asset can be listed as:
     *    name    asset/js/<name>.js
     *   ./file   full path to file, relative to plugin root
     *   /file    full path to file, relative to WordPress root
     *   url      full URL to file
     */
    protected const SCRIPTS = [];

    /**
     * Admin Pages
     *
     * Entry format:
     *    pageSlug => pageClass
     */
    protected const ADMIN_PAGES = [];

    /**
     * Admin Menus
     *
     * Entry formats:
     *    +menuSlug[@pos] => pageSlug
     *    pageSlug[@pos] => [subpageSlug, ...]
     *
     * Using +<menuSlug> will add the page to an existing menu entry,
     * while using <pageSlug> will add the page as a new top level
     * menu with one or more submenu entries for the other pages. If
     * an `@pos` is used, it specifies the position.
     */
    protected const ADMIN_MENUS = [];

    /**
     * Admin Actions
     *
     * Entry format:
     *    actionSlug => className
     */
    protected const ADMIN_ACTIONS = [];

    /**
     * Settings Groups
     *
     * Entry formats:
     *    name => className
     *
     * Name must be a lowercase alphanumeric identifier (can contain '_')
     * and className must be a Settings class.
     */
    public const SETTINGS_GROUPS = [];

    /**
     * Shortcodes
     *
     * Entry formats:
     *    tag => className                  calls <class>::render()
     *    tag => [className, methodName]    calls <class>::<method>()
     *    tag => ['new', className]         calls (new <class>())()
     */
    public const SHORTCODES = [];

    /**
     * Background jobs
     *
     * Entry format:
     *    name => [period, className]
     *
     * Name must be a lowercase alphanumeric identifier (can contain '_')
     * Built-in periods: 'hourly' | 'twicedaily' | 'daily' | 'weekly'
     * Use '' for period to skip scheduling but allow manual triggering
     *
     * Custom periods can be added by overriding customPeriods() below
     */
    public const JOBS = [];

    /** List of obsolete background jobs to be removed on upgrade */
    public const OLD_JOBS = [];

    /**
     * @internal Class properties exposed by singleton instance
     */
    private const PROPERTIES = [
        'name'        => 'name',
        'path'        => 'path',
        'slug'        => 'slug',
        'namespace'   => 'namespace',
        'text_domain' => 'textDomain',
        'url'         => 'url',
        'version'     => 'version',
    ];


    // ---------------------------------------------------------------------
    // Class properties
    // ---------------------------------------------------------------------
    // The plugin metadata is handled internally by the framework, but are
    // available to the child class via public accessor methods, such as
    // self::slug(), self::path(), etc. These accessors are also available
    // to other plugin code using Plugin::<name>() format.

    // Plugin metadata
    private static string $slug;            // self::slug()
    private static string $name;            // self::name()
    private static string $version;         // self::version()
    private static string $textDomain;      // self::textDomain()
    private static string $namespace;       // self::namespace()
    private static string $path;            // self::path()
    private static string $url;             // self::url()

    // Runtime state (internal use only)
    private static string $loader;              // bootstrap loader
    private static ?self $instance = null;      // singleton instance


    // ---------------------------------------------------------------------
    // Primary extension points
    // ---------------------------------------------------------------------
    // These are the core lifecycle hooks the plugin subclass should
    // implement to initialize features, register admin components, etc.

    /** Initialize plugin features */
    protected static function initialize(): void {}

    /** Initialize plugin admin features */
    protected static function initializeAdmin(): void {}

    /** Triggered on plugin activation */
    protected static function activate(): void {}

    /** Triggered on plugin deactivation */
    protected static function deactivate(): void {}

    /** Triggered after plugin upgrade completes */
    protected static function finishUpgrade(): void {}


    // ---------------------------------------------------------------------
    // Optional extension point
    // ---------------------------------------------------------------------
    // These methods have functional defaults but can be customized for
    // special cases (e.g. custom asset or shortcode loading).

    /**
     * Called just after initialize() to register shortcodes
     *
     * Default behavior is to register all entries defined by the plugin in
     * the SHORTCODES constant.
     */
    protected static function addShortcodes(): void
    {
        $rawContent = [];
        foreach (static::SHORTCODES as $tag => $handler) {
            add_shortcode($tag, [ShortcodeHandler::class, 'handler']);
            if (
                ($handler[0] ?? null) === 'new'
                && class_exists($class = $handler[1] ?? null)
                && $class::RAW_CONTENT
            ) {
                $rawContent[] = $tag;
            }
        }
        if ($rawContent) Shortcode::registerRawContent($rawContent);
    }

    /**
     * Used to conditionally include Frontend CSS/JS assets
     *
     * When the `wp_enqueue_scripts` action occurs, all assets defined in
     * STYLESHEETS and SCRIPTS will be registered with WordPress. If any
     * have been deferred, use Assets::include(<name>) to load them on
     * the current page. This MUST be done from addFrontendAssets() or
     * addAdminAssets().
     *
     * Note: General plugin assets (i.e. those defined without an alias)
     * will already be included for frontend pages.
     *
     * Example:
     *    The following will only add the plugin's CSS/JS assets defined
     *    with the 'extra' name on a page/post with a specific shortcode.
     *        if (Context::isPostWithShortcode('the-shortcode')) {
     *            Assets::include('extra');
     *        }
     *
     * It is recommended you only include assets if actually required
     * to improve frontend performance.
     *
     * @see Plugin::addAdminAssets() for including them on admin pages
     */
    protected static function addFrontendAssets(): void {}

    /**
     * Used to include Admin CSS/JS assets
     *
     * When the `admin_enqueue_scripts` action occurs, this function can
     * be used to load assets. By default, no assets are automatically
     * included on admin pages. Below are two use case examples.
     *
     * Use case 1: If the general frontend assets are needed when in the
     * block editor preview for posts, you can call Assets::include()
     *    if (
     *        Context::isBlockEditor()
     *        && Context::isAdminPage('post', 'post-new')
     *    ) {
     *        Assets::include();
     *    }
     *
     * Use case 2: If admin-specific assets are needed they can be added
     * to STYLESHEETS and SCRIPTS with a name such as 'admin', and
     * included on the appropriate admin pages with:
     *    if (Context::isAdminPage('somePage')) {
     *        Assets::include('admin');
     *    }
     *
     * It is recommended you only include assets when actually required
     * and you use the Context::isAdminPage(name,...) helper to limit
     * them to the correct admin pages where they are needed.
     */
    protected static function addAdminAssets(): void {}

    /**
     * Used to add all admin menus and submenus
     */
    private static function addAdminMenus(): void
    {
        foreach (static::ADMIN_MENUS as $key => $value) {
            if (($i = strpos($key, '@')) === false) {
                $pos = null;
            } else {
                $pos = 0 + substr($key, $i + 1);
                $key = substr($key, 0, $i);
            }
            if (substr($key, 0, 1) == '+') {
                // +menuSlug[@pos] => pageSlug
                $menuSlug = substr($key, 1);
                $pageSlug = $value;
                AdminPage::addSubmenuEntry($pageSlug, $menuSlug, $pos);
            } else {
                // pageSlug[@pos] => [subpageSlug, ...]
                $pageSlug = $key;
                $subpageSlugs = $value;
                AdminPage::addNewMenu($pageSlug, $pos);
                foreach ($subpageSlugs as $subpageSlug) {
                    AdminPage::addSubmenuEntry($subpageSlug, $pageSlug);
                }
            }
        }
    }

    /**
     * Used to get localized admin notice text
     *
     * @return string notice to show to user after an admin action completes
     */
    protected static function adminNotice(string $code): ?string { return null; }

    /**
     * Used to get additional action links to show in plugins page under plugin name
     *
     * WordPress shows Activate/Deactivate/Delete links by default. This function
     * allows additional items such as "Settings" to be shown. Each entry returned
     * should be a short link with a unique key to identify it. Normally, entries are
     * added to the start of the actions. Prefix the key with "+" to add to after
     * all other keyed entries (will attempt to add before any non-keyed entries).
     *
     * @return array<string,string> action links to add (key => link)
     */
    protected static function pluginsActionLinks(): array { return []; }

    /**
     * Used to get additional meta data to show in plugins page under description
     *
     * WordPress shows plugin version, author with link), and a "View Details" link
     * by default. This function allows additional items to be shown. Links can be
     * included to internal or external pages as needed.
     *
     * @return string[] text/links to be added
     */
    protected static function pluginsMetaData(): array { return []; }

    /**
     * Define custom periods for background jobs
     *
     * If a custom time period is needed for use with any of the plugin's
     * background jobs, this function should be implemented and return
     * the periods in the format required by WordPress. Each entry must
     * have an `interval` in seconds and a `display` value which should be
     * translated with __() and the literal value of plugin's text domain.
     *
     * Note: will only be used if entries exist in JOBS constant
     *
     * Examples of typical custom entries:
     *   'every_five_minutes'
     *       => ['interval' => 300, 'display' => __('Every 5 Minutes', <textDomain>)],
     *   'every_fifteen_minutes'
     *       => ['interval' => 900, 'display' => __('Every 15 Minutes', <textDomain>)],
     *   'every_thirty_minutes'
     *       => ['interval' => 1800, 'display' => __('Every 30 Minutes', <textDomain>)],
     *   'every_two_hours'
     *       => ['interval' => 7200, 'display' => __('Every 2 Hours', <textDomain>)],
     *   'every_two_days'
     *       => ['interval' => 172800, 'display' => __('Every 2 Days', <textDomain>)],
     *   'monthly'
     *       => ['interval' => 2592000, 'display' => __('Monthly', <textDomain>)],
     */
    protected static function customPeriods(): array { return []; }

    // Instance property accessors
    final public function __get(string $name)
    {
        return isset(self::PROPERTIES[$name]) ? self::$$name : null;
    }
    final public function __isset(string $name): bool
    {
        return isset(self::PROPERTIES[$name]);
    }


    // ---------------------------------------------------------------------
    // Public interface
    // ---------------------------------------------------------------------

    /**
     * Initialize plugin
     *
     * Must be called by bootstrap loader to initialize plugin.
     *
     * @param   string  $name           plugin name
     * @param   string  $version        plugin version
     * @param   string  $slug           plugin slug
     * @param   string  $textDomain     localization text domain ('' = none)
     * @param   string  $namespace      root namespace of plugin
     * @param   string  $loader         bootstrap loader file
     */
    final public static function init(
        string $name,
        string $version,
        string $slug,
        string $textDomain,
        string $namespace,
        string $loader
    ): void {
        self::$name         = $name;
        self::$version      = $version;
        self::$slug         = $slug;
        self::$textDomain   = $textDomain;
        self::$namespace    = $namespace;
        self::$loader       = $loader;
        self::$path         = plugin_dir_path($loader);
        self::$url          = plugin_dir_url($loader);

        self::registerAutoloader($namespace);
        self::registerHooks();
    }

    // Class property accessors
    final public static function slug(): string         { return self::$slug; }
    final public static function name(): string         { return self::$name; }
    final public static function version(): string      { return self::$version; }
    final public static function textDomain(): string   { return self::$textDomain; }
    final public static function namespace(): string    { return self::$namespace; }
    final public static function path(): string         { return self::$path; }
    final public static function url(): string          { return self::$url; }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------

    // --- Autoload and registration ---

    // Register class autoloader
    private static function registerAutoloader(string $namespace): void
    {
        $prefix = $namespace . '\\';
        $length = strlen($prefix);
        $path   = self::$path . 'includes/';
        spl_autoload_register(function ($class) use ($prefix, $length, $path) {
            if (strncmp($class, $prefix, $length) === 0) {
                $relative = substr($class, $length);
                $file = $path . str_replace('\\', '/', $relative) . '.php';
                if (is_file($file)) require_once $file;
            }
        });
    }

    // Register WordPress hooks
    private static function registerHooks(): void
    {
        register_activation_hook(self::$loader, static function () {
            static::activate();
            JobDispatcher::installAllJobs();
        });
        register_deactivation_hook(self::$loader, static function () {
            JobDispatcher::removeAllJobs();
            static::deactivate();
        });
        add_action('upgrader_process_complete', static function ($upgrader, $info) {
            if (
                $upgrader instanceof Plugin_Upgrader
                && ($info['action'] ?? '') === 'update'
                && ($info['type'] ?? '') === 'plugin'
            ) {
                $match = plugin_basename(self::$loader);
                foreach ($info['plugins'] ?? [] as $plugin) {
                    if ($plugin === $match) {
                        JobDispatcher::removeAllJobs();
                        static::finishUpgrade();
                        JobDispatcher::installAllJobs();
                        break;
                    }
                }
            }
        }, 10, 2);
        add_action('init', static function () {
            static::initialize();
            static::addShortcodes();
        });
        add_action('admin_init', static function () {
            if ($GLOBALS['pagenow'] ?? '' === 'options.php') {
                foreach (static::SETTINGS_GROUPS as $name => $_class) {
                    Settings::register($name);
                }
            }
            self::addPluginsFilters();
            self::defineAdminPages();
            self::registerAdminActions();
            static::initializeAdmin();
        });
        if (static::JOBS) {
            add_action(self::jobDispatchHook(), [JobDispatcher::class, 'dispatch']);
            add_filter('cron_schedules', static function ($schedule) {
                return $schedule + static::customPeriods();
            });
        }
        add_action('wp_enqueue_scripts', static function () {
            wp_register_script(
                // Shared framework handle prefix (ISGDev) to avoid duplicating "js" class injection across plugins
                // phpcs:ignore WordPress.NamingConventions.ValidHookName.NotPrefixed -- shared framework handle
                'isgdev-head-js-flag',
                false,  // no source file for inline-only script
                [],     // no dependencies
                null,   // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion -- no version needed if inline
                false   // no additional args
            );
            wp_enqueue_script('isgdev-head-js-flag');
            wp_add_inline_script(
                'isgdev-head-js-flag',
                'document.documentElement.classList.add("js");',
                'before'
            );
            self::defineAssets(true);
            static::addFrontendAssets();
        });
        add_action('admin_enqueue_scripts', static function () {
            self::addCoreAdminAssets();
            self::defineAssets();
            static::addAdminAssets();
            wp_enqueue_style('wp-color-picker');
            wp_register_script(
                // Shared framework handle prefix (ISGDev) to avoid duplicating code across my plugins
                // phpcs:ignore WordPress.NamingConventions.ValidHookName.NotPrefixed -- shared framework handle
                'isgdev-color-picker',
                false,                  // no source file for inline-only script
                ['wp-color-picker'],    // dependency on WP color picker
                null,   // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion -- no version needed if inline
                true    // load in footer
            );
            wp_enqueue_script('isgdev-color-picker');
            wp_add_inline_script(
                'isgdev-color-picker',
                'jQuery(document).ready(function($){ $(\'.isgdev-color-field\').wpColorPicker(); });',
                'before'
            );
        });
        add_action('admin_menu', static function () {
            self::defineAdminPages();
            static::addAdminMenus();
        });
        add_action('admin_notices', static fn () => static::showAdminNotices());
    }

    // --- Internal helpers ---

    /** Display unauthorized error and exit */
    final public static function unauthorized(): void
    {
        wp_die(esc_html(static::__('unauthorized')));
    }

    /** Plugin text translation helper */
    final public static function __(string $word): string
    {
        static $helper = null;
        $helper ??= class_exists($class = self::$namespace . '\PluginText')
            ? $class
            : self::$namespace . '\Core\Base\PluginText';
        return $helper::{$word . 'Text'}();
    }

    /** Add filters for plugins page */
    private static function addPluginsFilters(): void
    {
        $pluginBasename = plugin_basename(self::$loader);
        add_filter(
            'plugin_action_links_' . $pluginBasename,
            function ($links) {
                $prepend = [];
                $append = [];
                foreach (static::pluginsActionLinks() as $key => $entry) {
                    if (substr($key, 0, 1) === '+') {
                        $append[substr($key, 1)] = wp_kses_post($entry);
                        unset($prepend[$key]);
                    } else {
                        $prepend[$key] = wp_kses_post($entry);
                    }
                }
                if ($append) {
                    $intKeys = array_filter(array_keys($links), 'is_int');
                    if (empty($intKeys)) {
                        $links += $append;
                    } else {
                        $i = key($intKeys);
                        $links = array_slice($links, 0, $i, true) + $append + array_slice($links, $i, null, true);
                    }
                }
                return $prepend + $links;
            }
        );
        add_filter('plugin_row_meta', function ($links, $file, $data, $status) use ($pluginBasename) {
            if ($file === $pluginBasename) {
                foreach (static::pluginsMetaData() as $entry) {
                    $links[] = wp_kses_post($entry);
                }
            }
            return $links;
        }, 10, 4);
    }

    /** Define all admin pages and setting groups */
    private static function defineAdminPages(): void
    {
        static $done = false;
        if (!$done) {
            foreach (static::ADMIN_PAGES as $slug => $class) {
                AdminPage::add(new $class($slug));
            }
            $done = true;
        }
    }

    /** Register callbaacks for all admin actions */
    private static function registerAdminActions(): void
    {
        foreach (static::ADMIN_ACTIONS as $actionSlug => $actionClass) {
            add_action(
                'admin_post_' . self::$slug . '_' . $actionSlug,
                fn () => $actionClass::handle($actionSlug)
            );
        }
    }

    /** */
    private static function showAdminNotices(): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only msg code
        $code = isset($_GET['admin_msg']) && is_string($_GET['admin_msg']) ? sanitize_key(wp_unslash($_GET['admin_msg'])) : null;

        if ($code && ($text = static::adminNotice($code))) {
            printf('<div class="notice notice-success is-dismissible"><p>%s</p></div>', esc_html($text));
        }
    }

    /** Normalized cron hook name used by registerHooks() and the Job dispatcher */
    final public static function jobDispatchHook(): string
    {
        static $name = null;
        return $name ?? ($name = str_replace('-', '_', self::$slug) . '_job_dispatch');
    }

    /** Register and/or auto-enqueue assets */
    private static function defineAssets(bool $inclGeneral = false): void
    {
        static $done = false;
        if (!$done) {
            // first-time registration and optional enqueuing (normal invocation)
            foreach (static::STYLESHEETS as $name => $asset) {
                self::defineAsset('css', $name, $asset, $inclGeneral);
            }
            foreach (static::SCRIPTS as $name => $asset) {
                self::defineAsset('js', $name, $asset, $inclGeneral);
            }
        } elseif ($inclGeneral) {
            // edge case: specifically called again with enqueuing requested
            Assets::include();
        }
        $done = true;
    }
    private static function defineAsset($type, $name, $asset, $enqueue): void
    {
        if (is_int($name)) {
            $name = '';
        } else {
            $enqueue = false;
        }
        if (is_array($asset)) {
            foreach ($asset as $a) Assets::add($type, $name, $a, $enqueue);
        } else {
            Assets::add($type, $name, $asset, $enqueue);
        }
    }

    /** Enqueue core admin assets */
    private static function addCoreAdminAssets(): void
    {
        self::defineAsset('css', 0, './includes/Core/assets/admin.css', true);
    }

    // --- Singleton management ---

    /** Get singleton instance */
    final public static function instance(): Plugin
    {
        if (!self::$instance) self::$instance = new static();
        return self::$instance;
    }

    // Enforce as singleton
    private function __construct() {}
    private function __clone() { }
    final public function __wakeup() { throw new LogicException('Cannot unserialize singleton'); }
}
