<?php

namespace CocktailRecipes\Core\Base;

use CocktailRecipes\Plugin;
use CocktailRecipes\Core\Exceptions\InvalidJobDefinition;
use CocktailRecipes\Core\Helpers\Logger;
use CocktailRecipes\Core\Helpers\Text;
use CocktailRecipes\Core\Helpers\Time;
use InvalidArgumentException;
use Throwable;

abstract class Job
{
    // ---------------------------------------------------------------------
    // Methods to override in subclasses
    // ---------------------------------------------------------------------

    /**
     * Entry point called when job is executed (required)
     *
     * @return  bool    return true if job succeeds, false on failure
     */
    abstract public function handle(): bool;


    // ---------------------------------------------------------------------
    // Public interface for plugin use
    // ---------------------------------------------------------------------

    /** Get interval period for a named job period */
    final public static function periodTime(string $period): int
    {
        return self::periods()[$period]['interval'] ?? 0;
    }

    /** Get display text for a named job period */
    final public static function periodText(string $period): string
    {
        return self::periods()[$period]['display'] ?? $period;
    }

    /**
     * Manually execute a job
     *
     * @return  bool    returns true if job succeeds, false on failure
     */
    final public static function trigger(string $name): bool
    {
        try {
            if (!self::isValidJobName($name)) {
                throw new InvalidArgumentException('Invalid job name: ' . esc_html($name));
            }
            if (($info = Plugin::JOBS[$name] ?? null) === null) {
                throw new InvalidArgumentException('Unknown job name: ' . esc_html($name));
            }
            if (!($class = $info[1] ?? null) || !is_string($class) || !class_exists($class)) {
                throw new InvalidJobDefinition(esc_html($name));
            }
        } catch (Throwable $e) {
            Logger::error('Job trigger failed [' . get_class($e) . ']: ' . $e->getMessage());
            if (defined('WP_DEBUG') && WP_DEBUG) throw $e;
            return false;
        }
        return (new $class())($name);
    }

    /** Get list of scheduled jobs for the plugin */
    final public static function scheduledJobs(): array
    {
        $hook = Plugin::jobDispatchHook();
        $jobs = [];
        foreach (Plugin::JOBS as $name => $info) {
            if (!self::isValidJobName($name)) continue;
            if ($timestamp = wp_next_scheduled($hook, [$name])) {
                $jobs[] = [
                    'name'      => $name,
                    'period'    => $info[0],
                    'next_run'  => $timestamp,
                ];
            }
        }
        usort($jobs, fn($a, $b) => $a['next_run'] <=> $b['next_run']);
        return $jobs;
    }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------
    // Job definition format:
    //    [period, className]

    /** Install all background jobs into WordPress scheduler */
    final public static function installAllJobs(): void
    {
        $hook  = Plugin::jobDispatchHook();
        $names = [];
        $now   = time();
        foreach (Plugin::JOBS as $name => $info) {
            if (!self::isValidJobName($name)) {
                Logger::error("invalid job name defined: $name");
                continue;
            }
            if (!self::isValidJobDefinition($info)) {
                Logger::error("invalid job definition: $name");
                continue;
            }
            $period = $info[0];
            if (wp_next_scheduled($hook, [$name])) {
                $extra = ($oldPeriod = self::jobPeriod($name, $hook)) ? " ($oldPeriod)" : '';
                wp_clear_scheduled_hook($hook, [$name]);
                Logger::notice(($period ? 'rescheduling' : 'removed') . " job: $name$extra");
            }
            if ($period) {
                wp_schedule_event($now, $period, $hook, [$name]);
                Logger::info("scheduled job: $name ($period)");
            }
        }
    }

    /** Remove all background jobs from WordPress schedule */
    final public static function removeAllJobs(): void
    {
        $hook = Plugin::jobDispatchHook();
        $remove = function ($name) use ($hook) {
            if (!self::isValidJobName($name)) return false;
            if (!wp_next_scheduled($hook, [$name])) return false;
            wp_clear_scheduled_hook($hook, [$name]);
            return true;
        };
        foreach (Plugin::JOBS as $name => $info) {
            if ($remove($name)) {
                Logger::info("removed scheduled job: $name (" . self::periodText($info[0]) . ")");
            }
        }
        foreach (Plugin::OLD_JOBS as $name) {
            $period = self::jobPeriod($name, $hook);    // must save period (if it exists) first
            if ($remove($name)) {
                $extra = $period ? ' (' . self::periodText($period) . ')' : '';
                Logger::info("removed obsolete scheduled job: $name$extra");
            }
        }
    }

    /** True if job name is valid */
    private static function isValidJobName($name): bool
    {
        return is_string($name) && Text::isIdentifier($name) && strtolower($name) == $name;
    }

    /** True if job definition is valid */
    private static function isValidJobDefinition($info): bool
    {
        if (!is_array($info) || count($info) < 2) return false;
        [$period, $class] = $info;
        if (!is_string($period) || ($period && !isset(self::periods()[$period]))) return false;
        if (!is_string($class) || !class_exists($class, true)) return false;
        return true;
    }

    /** Get defined periods */
    private static function periods(): array
    {
        static $periods = null;
        return $periods ?? ($periods = wp_get_schedules());
    }

    /** Get period for existing job */
    private static function jobPeriod(string $name, ?string $hook = null): ?string
    {
        $period = wp_get_schedule($hook ?? Plugin::jobDispatchHook(), [$name]);
        return $period ?: null;
    }

    /** Runtime dispatcher */
    final public static function dispatch(string $name): void
    {
        if (
            self::isValidJobName($name)
            && ($class = Plugin::JOBS[$name][1] ?? null)
            && is_string($class)
            && class_exists($class)
        ) {
            (new $class())($name);
        } else {
            Logger::error("Cannot run undefined job: $name");
        }
    }

    /** Instance invocation */
    final public function __invoke(string $name): bool
    {
        $success = false;
        try {
            Logger::info("$name job started");
            $started = microtime(true);
            $success = $this->handle();
            $ended = microtime(true);
            Logger::info("$name job ended after " . Time::formatElapsed($ended - $started));
        } catch (Throwable $e) {
            $ended = microtime(true);
            $message = "$name job failed";
            if (isset($started)) {
                $message .= ' after ' . Time::formatElapsed($ended - $started);
            }
            Logger::error("$message [" . get_class($e) . ']: ' . $e->getMessage());
            if (defined('WP_DEBUG') && WP_DEBUG) throw $e;
        }
        return $success;
    }
}
