<?php

namespace CocktailRecipes\Core\Admin;

use CocktailRecipes\Core\Base\ReadOnlyProps;

/**
 * Abstract Admin Tab
 *
 * Base class for admin page tabs. Each tab handles its own settings
 * registration and rendering.
 *
 * @property string $slug   read-only slug
 * @property string $title  read-only localized tab title
 */
abstract class Tab extends ReadOnlyProps
{
    use HasForms;

    // ---------------------------------------------------------------------
    // Class properties
    // ---------------------------------------------------------------------
    // These properties are set for each page tab in the init() function.

    /** @var string tab title shown in navigation */
    protected string $title;


    // ---------------------------------------------------------------------
    // Internal properties
    // ---------------------------------------------------------------------
    // These properties are used within the class and should not be
    // modified directly

    /** @var string tab slug used in URL */
    protected string $slug;


    // ---------------------------------------------------------------------
    // Primary extension points
    // ---------------------------------------------------------------------
    // These are the extension points which an admin tab must implement


    /**
     * Initialize tab properties
     *
     * Properties which must set:
     *   $title     title shown in navigation bar
     */
    abstract public function init(): void;

    /**
     * Initialize tab content
     *
     * Use these to setup the tab content:
     *   addSettings()      registers a settings group to be used
     *   setButton()        set properties of submit button
     *   addSection()       creates an input section
     *   addInput()         adds a single input field
     *   addInputs()        adds a group of input fields
     *   addText()          adds text to show within form
     *
     * For read-only style tabs or other customized tabs, you can
     * call addSection() and addText() without requiring inputs.
     */
    abstract public function initContent(): void;


    // ---------------------------------------------------------------------
    // Optional extension points
    // ---------------------------------------------------------------------
    // These extension points can be overridden as needed

    /**
     * Render tab content
     */
    public function render(): void
    {
        if ($this->intro) $this->renderIntro();
        foreach ($this->forms as $id => $settings) {
            if ($settings === null) {
                do_settings_sections($id);
            } else {
                echo '<form method="post" action="options.php">';
                settings_fields($settings::OPTION_NAME);
                do_settings_sections($id);
                $button = $this->buttons[$id] ?? [];
                if ($warning = $button['warning'] ?? null) {
                    echo '<p class="description btn-warning">' . wp_kses_post($warning) . '</p>';
                }
                submit_button(
                    $button['text'] ?? '',
                    $button['type'] ?? 'primary'
                );
                echo '</form>';
            }
        }
    }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------

    final public function __construct(string $slug)
    {
        $this->slug = $slug;
    }
}
