<?php

namespace CocktailRecipes\Core\Admin;

use CocktailRecipes\Core\Exceptions\UnknownSettingsGroup;
use CocktailRecipes\Plugin;

/**
 * Abstract Admin Settings Group
 *
 * Base class for creating an admin settings group
 */
abstract class Settings
{
    // ---------------------------------------------------------------------
    // Class properties
    // ---------------------------------------------------------------------
    // These properties are set for each admin page in the init() function

    /**
     * WordPress option name
     *
     * Set this to a unique name, namespaced for the plugin. It
     * will be used for the option database entry by WordPress to store all
     * of the settings within this group, as an array.
     */
    public const OPTION_NAME = '';

    /**
     * Setting fields within the group
     *
     * Use this to list all setting fields within the group. Each entry is
     * the setting name and the associated type, or an array with all the
     * the properties for the setting.
     *
     * Examples
     *   'enabled' => 'bool',
     *   'use_color' => [
     *       'type'    => 'enum',
     *       'values'  => ['on', 'off'],
     *       'default' => 'off',
     *   ],
     *   'max_rows' => [
     *       'type' => 'num',
     *       'min' => 1,
     *       'max' => 10
     *   ]
     *
     * Types
     *   bool       true/false value
     *   enum       value selected from a fixed list of values
     *   num        numeric value (integer)
     *   text       short string value (whitespace & line breaks stripped)
     *   url        a URL
     *   email      an email address
     *   color      an HTML hex color
     *   textarea   text block (whitespace & line breaks preserved)
     *   markdown   markdown block
     *
     * Properties
     *   type       string  one of the data types above (required)
     *   default    mixed   optional default value based on type
     *   values     array   allowed values (required for enum; optional for num)
     *   min        int     minimum for a num type (not used if values defined)
     *   max        int     maximum for a num type (not used if values defined)
     *   step       int     step value for num type; i.e. min, min+step, ...
     *   minlen     int     minimum length of a text type (invalid if too short)
     *   maxlen     int     maximum length of text/textarea/markdown (truncated)
     *   empty      bool    allow empty values for text-like fields (default true)
     *   pattern    string  regex for text type (no delimiters or anchors)
     */
    protected const FIELDS = [];


    // ---------------------------------------------------------------------
    // Internal properties
    // ---------------------------------------------------------------------

    // All settings group instances
    private static array $groups = [];

    // classname to group map
    private static ?array $classes = null;

    // Current field values
    protected $data = null;


    // ---------------------------------------------------------------------
    // Optional extension points
    // ---------------------------------------------------------------------

    /**
     * Custom field validation hook for child classes
     *
     * Override this method to add additional validation logic specific
     * to the settings group. Called after standard sanitization.
     *
     * @param array $output Sanitized field values
     * @return array        Modified field values
     */
    protected function validateFields(array $output): array
    {
        return $output;
    }


    // ---------------------------------------------------------------------
    // Public interfaces
    // ---------------------------------------------------------------------

    /**
     * Admin page registration and return of the Settings object
     *
     * @param  string $name     settings group name from Plugin::SETTINGS_GROUP
     */
    final public static function register(string $name): Settings
    {
        $settings   = self::group($name);
        $optionName = $settings::OPTION_NAME;
        register_setting($optionName, $optionName, [
            'type' => 'array',
            'default' => [],
            'sanitize_callback' => [$settings, 'sanitize'],
        ]);
        return $settings;
    }

    /**
     * Get a settings group
     *
     * @param  string $name     settings group name from Plugin::SETTINGS_GROUP
     * @return object           Settings object for the settings group
     */
    final public static function group(string $name): Settings
    {
        if (!isset(self::$groups[$name])) {
            $class = Plugin::SETTINGS_GROUPS[$name] ?? null;
            if (!$class) throw new UnknownSettingsGroup(esc_html($name));
            self::$groups[$name] = new $class();
        }
        return self::$groups[$name];
    }

    /**
     * Get a setting field from a group
     *
     * Can be called as Settings::field(group,field) or SettingsChild::(field)
     *
     * @param string $groupOrField  settings group name or field name
     * @param string $field         field name
     * @return mixed|null           field value or default; null if field not defined
     */
    final public static function field(string $groupOrField, ?string $field = null)
    {
        return static::class === self::class
            ? self::group($groupOrField)->get($field)
            : self::all()->get($groupOrField);
    }

    /** Get settings group for a child Settings class */
    final public static function all(): Settings
    {
        if (self::$classes === null) self::$classes = array_flip(Plugin::SETTINGS_GROUPS);
        return self::group(self::$classes[static::class]);
    }

    /**
     * Get a specific setting value or its default value
     *
     * @param  string $field    field name
     * @return mixed|null       field value or default; null if field not defined
     */
    final public function get(string $field)
    {
        return ($properties = $this->properties($field))
            ? $this->value($field, $properties)
            : null;
    }

    /**
     * Get field properties and value for admin field rendering
     *
     * @param  string $field    field name
     * @return mixed            field data (properties + value); null if not defined
     */
    final public function fieldData(string $field)
    {
        if ($fieldData = $this->properties($field)) {
            $fieldData['value'] = $this->value($field, $fieldData);
            $fieldData['isset'] = isset($this->data[$field]);
        }
        return $fieldData;
    }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------

    // Get field properties (array) or null if not defined
    private function properties(string $name): ?array
    {
        $properties = static::FIELDS[$name] ?? null;
        if (is_string($properties)) $properties = ['type' => $properties];
        return $properties;
    }

    // Get field value or its default
    private function value(string $field, array $properties)
    {
        if ($this->data === null) {
            $this->data = get_option(static::OPTION_NAME, []);
        }
        return $this->data[$field]
            ?? $properties['default']
            ?? $this->typeDefault($properties['type'] ?? '');
    }

    // Get default based on field type
    private function typeDefault(string $type)
    {
        switch ($type) {
            case 'bool': return false;
            case 'num': return 0;
        }
        return '';
    }

    // WordPress sanitization callback
    final public function sanitize($input) {

        $output = get_option(static::OPTION_NAME, []);
        if (!is_array($input)) return $output;

        foreach (static::FIELDS as $name => $properties) {
            $val = $input[$name] ?? null;
            if (is_string($properties)) $properties = ['type' => $properties];
            switch ($properties['type'] ?? '') {

                case 'bool':
                    if ($val === null && empty($input['ck__' . $name])) continue 2;
                    $val = !empty($val);
                    break;

                case 'enum':
                    if ($val === null || !in_array($val, $properties['values'] ?? [], true)) continue 2;
                    break;

                case 'num':
                    if ($val === null || !preg_match('/^[-+]?[0-9]+$/', $val)) continue 2;
                    $val += 0;
                    if ($vals = $properties['values'] ?? null) {
                        if (!in_array($val, $vals)) continue 2;
                    } elseif (($min = $properties['min'] ?? null) !== null && $val < $min) {
                        $val = $min;
                    } elseif (($max = $properties['max'] ?? null) !== null && $val > $max) {
                        $val = $max;
                    }
                    break;

                case 'text':
                    if ($val === null) continue 2;
                    $val = sanitize_text_field($val);
                    if (($min = $properties['minlen'] ?? null) !== null && strlen($val) < $min) continue 2;
                    if (($max = $properties['maxlen'] ?? null) !== null && strlen($val) > $max) {
                        $val = rtrim(substr($val, 0, $max));
                    }
                    if (trim($val) == '' && !($properties['empty'] ?? true)) continue 2;
                    if (
                        ($regex = $properties['pattern'] ?? null) !== null
                        && !preg_match('/^' . $regex . '$/', $val)
                    ) {
                        continue 2;
                    }
                    break;

                case 'email':
                    if ($val === null || !is_email($val)) continue 2;
                    if ($val == '' && !($properties['empty'] ?? true)) continue 2;
                    break;

                case 'url':
                    if ($val === null || esc_url_raw($val) !== $val) continue 2;
                    if ($val == '' && !($properties['empty'] ?? true)) continue 2;
                    break;

                case 'color':
                    if ($val === null || ($val = sanitize_hex_color($val)) === null) continue 2;
                    break;

                case 'textarea':
                    if ($val === null) continue 2;
                    $val = sanitize_textarea_field($val);
                    if (($max = $properties['maxlen'] ?? null) !== null && strlen($val) > $max) {
                        $val = substr($val, 0, $max);
                    }
                    if (trim($val) == '' && !($properties['empty'] ?? true)) continue 2;
                    break;

                case 'markdown':
                    if ($val === null) continue 2;
                    $val = $this->restoreMarkdown(wp_kses_post($this->preserveMarkdown($val)));
                    if (($max = $properties['maxlen'] ?? null) !== null && strlen($val) > $max) {
                        $val = substr($val, 0, $max);
                    }
                    if (trim($val) == '' && !($properties['empty'] ?? true)) continue 2;
                    break;
            }
            $output[$name] = $val;
        }

        return $this->validateFields($output);
    }

    // Protect specific markdown sequences from being stripped by wp_kses_post()
    private function preserveMarkdown(string $text): string
    {
        #$text = preg_replace('/[\x00-\x08\x0E-\x1F\x7F]/', '', $text);
        $text = preg_replace('/<(([a-z][a-z0-9+.-]*):([^ >]+))>/i', "@=={\\1}==@", $text);
        return preg_replace(
            '/<([A-Za-z0-9._%+\-]+@[A-Za-z0-9.\-]+\.[A-Za-z]{2,})>/',
            "@=={\\1}==@",
            $text
        );
    }

    // Restore markdown sequences protected by preserveMarkdown()
    private function restoreMarkdown(string $text): string
    {
        return strtr($text, ["@=={" => '<', "}==@" => '>']);
    }

}
