<?php

namespace CocktailRecipes\Core\Admin;

if (!defined('ABSPATH')) exit;

use CocktailRecipes\Plugin;

/**
 * Abstract Admin Page
 *
 * Base class for creating admin settings pages with optional tab support
 */
abstract class Page
{
    use HasForms;

    // ---------------------------------------------------------------------
    // Class properties
    // ---------------------------------------------------------------------
    // These properties are set for each admin page in the init() function.

    /**
     * Page title shown in browser/admin bar (required) */
    protected string $title;

    /** Menu title shown in admin menu (required) */
    protected string $menuTitle;

    /**
     * Menu icon
     *
     * For use with top-level menu pages to add an icon to the sidebar.
     *
     * Value types expected:
     *   base64-encoded SVG - e.g. 'data:image/svg+xml;base64,...
     *   Dashicons helper class - e.g. 'dashicons-chart-pie'
     *   'none' to allow icon to be added via CSS
     *
     * @see https://developer.wordpress.org/reference/functions/add_menu_page/
     */
    protected string $menuIcon = '';

    /**
     * Submenu title shown in admin
     *
     * For use with top-level menu pages to adjust menu text shown as the
     * first submenu entry, which always goes to the same location as the
     * top level menu. Defaults to same value as $this->title.
     */
    protected ?string $submenuTitle = null;

    /**
     * Required permissions (i.e. WordPress capability) to access page
     *
     * @see https://wordpress.org/documentation/article/roles-and-capabilities/
     */
    protected string $reqPerm = 'manage_options';


    // ---------------------------------------------------------------------
    // Internal properties
    // ---------------------------------------------------------------------
    // These properties are used within the class and should not be
    // modified directly

    /** @var array<string,string> menu load hooks installed (hook => page slug) */
    private static array $menuLoadHooks = [];

    /** @var array<string,Page> all admin pages (page slug => instance) */
    private static array $pages = [];

    /** @var string page slug used in URL */
    protected string $slug;

    /** @var array<string,Tab> all tabs for page (tag slug => tab) */
    protected array $tabs = [];

    /** @var ?string active tab slug */
    protected ?string $tab = null;

    /** @var string[] page footer content */
    protected array $pageFooters = [];


    // ---------------------------------------------------------------------
    // Primary extension points
    // ---------------------------------------------------------------------
    // These are the extension points which an admin page must implement

    /**
     * Initialize page proprties
     *
     * Properties which must set:
     *   $title             title shown in browser/admin bar
     *   $menuTitle         title shown in admin menu
     *
     * Properties which can optionally be set:
     *   $menuIcon          sidebar icon for a top level menu
     *   $submenuTitle      first submenu entry for a top level menu
     *   $reqPerm           user permissions required
     */
    abstract protected function init(): void;

    /**
     * Initialize page content
     *
     * If using tabs, use these functions:
     *   setTabs()          sets the tabs for the page
     *   addFooter()        adds optional footer to the page
     *
     * If not using tabs, use these to setup the page content:
     *   addSettings()      registers a settings group to be used
     *   setButton()        set properties of submit button
     *   addSection()       creates an input section
     *   addInput()         adds a single input field
     *   addInputs()        adds a group of input fields
     *   addText()          adds text to show within form
     *   addFooter()        adds optional footer to the page
     *
     * For read-only style pages or other customized pages, you can
     * call addSection() and addText() without requiring inputs.
     */
    abstract protected function initContent(): void;


    // ---------------------------------------------------------------------
    // Optional extension points
    // ---------------------------------------------------------------------
    // These extension points can be overridden as needed

    /**
     * Render page content
     *
     * For pages without tabs, by default this renders a form with the
     * associated settings group, sections and fields, along with the
     * submit button. If multiple setting groups are used, each will be
     * output as a separate form. Override only if more complex or fully
     * custom rendering is needed.
     *
     * For pages with tabs, this rendering should normally be left as-is
     * and the render() function of each tab can be overridden if needed.
     */
    protected function render(): void
    {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html($this->title); ?></h1>
            <?php if ($this->tabs): ?>
                <nav class="nav-tab-wrapper">
                    <?php foreach ($this->tabs as $slug => $tab): ?>
                        <a href="?page=<?php echo esc_attr($this->slug); ?>&tab=<?php echo esc_attr($slug); ?>"
                            class="nav-tab <?php echo $slug === $this->tab ? 'nav-tab-active' : ''; ?>">
                            <?php echo esc_html($tab->title); ?>
                        </a>
                    <?php endforeach; ?>
                </nav>
                <div class="tab-content">
                    <?php $this->tabs[$this->tab]->render(); ?>
                </div>
            <?php else: ?>
                <?php if ($this->intro) $this->renderIntro() ?>
                <?php foreach ($this->forms as $id => $settings): ?>
                    <?php if ($settings === null): ?>
                        <?php do_settings_sections($id); ?>
                    <?php else: ?>
                        <form method="post" action="options.php">
                            <?php
                                settings_fields($settings::OPTION_NAME);
                                do_settings_sections($id);
                                $button = $this->buttons[$id] ?? [];
                                if ($warning = $button['warning'] ?? null) {
                                    echo '<p class="description">' . wp_kses_post($warning) . '</p>';
                                }
                                submit_button(
                                    $button['text'] ?? '',
                                    $button['type'] ?? 'primary'
                                );
                            ?>
                        </form>
                    <?php endif; ?>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        <?php if ($this->pageFooters): ?>
            <div class="page-footer isgdev-admin-footer">
                <?php foreach ($this->pageFooters as $footer): ?>
                    <p><?php echo wp_kses_post($footer); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <?php
    }


    // ---------------------------------------------------------------------
    // Public interfaces
    // ---------------------------------------------------------------------

    /**
     * Get URL to a specific page in the admin console
     *
     * @param string $path  Admin file path; e.g. 'options-general.php', 'admin.php'
     * @param string $page  Page query parameter
     * @param string $tab   Tab query parameter
     * @return string       Full admin URL with query parameters
     */
    final public static function adminUrl(string $path, string $page = '', string $tab = ''): string
    {
        $url = admin_url($path);
        if ($page) {
            $url = add_query_arg('page', $page, $url);
            if ($tab) $url = add_query_arg('tab', $tab, $url);
        }
        return $url;
    }

    /**
     * Set tabs for the page
     *
     * @param array<Tab>    tabs for the page
     */
    final protected function setTabs(array $tabs): void
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only tab selection
        $requested = isset($_GET['tab']) && is_string($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : '';
        foreach ($tabs as $tab) {
            $tab->init();
            $this->tabs[$tab->slug] = $tab;
            if ($tab->slug == $requested) {
                $tab->initContent();
                $this->tab = $tab->slug;
            }
        }
        if (!$this->tab && $tabs) {
            $active = $tabs[0];
            $this->tab = $active->slug;
            $active->initContent();
        }
    }

    /** Add footer content to display at bottom of page */
    final protected function addFooter(string ...$text): void
    {
        array_push($this->pageFooters, ...$text);
    }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------

    // Called by Plugin to add each admin page
    final public static function add(Page $page): void
    {
        if (!isset(self::$pages[$page->slug])) {
            $page->init();
            self::$pages[$page->slug] = $page;
        }
    }

    /**
     * Called by Plugin to add top menu entries
     *
     * @param int|float|null $menuPos
     */
    final public static function addNewMenu(string $pageSlug, $menuPos = null): void
    {
        if ($page = self::$pages[$pageSlug] ?? null) {
            $hook = add_menu_page(
                $page->title,
                $page->menuTitle,
                $page->reqPerm,
                $pageSlug,
                [$page, 'renderPage'],
                $page->menuIcon,
                $menuPos
            );
            self::addMenuLoadHook($hook, $page);
            if ($page->submenuTitle) add_submenu_page(
                $pageSlug,
                $page->title,
                $page->submenuTitle,
                $page->reqPerm,
                $pageSlug,
                [$page, 'renderPage']
            );
        }
    }

    /**
     * Called by Plugin to add submenu entries
     *
     * @param int|float|null $menuPos
     */
    final public static function addSubmenuEntry(string $pageSlug, string $parentSlug, $menuPos = null): void
    {
        if ($page = self::$pages[$pageSlug] ?? null) {
            $hook = add_submenu_page(
                $parentSlug,
                $page->title,
                $page->menuTitle,
                $page->reqPerm,
                $pageSlug,
                [$page, 'renderPage'],
                $menuPos
            );
            self::addMenuLoadHook($hook, $page);
        }
    }

    private static function addMenuLoadHook(string $hook, Page $page)
    {
        if (!isset(self::$menuLoadHooks[$hook])) {
            add_action('load-' . $hook, function () use ($page) {
                $page->initContent();
            });
            self::$menuLoadHooks[$hook] = $page->slug;
        }
    }

    // Constructor
    final public function __construct(string $slug)
    {
        $this->slug = $slug;
    }

    // render the page if user has correct rights
    final public function renderPage(): void
    {
        if (!current_user_can($this->reqPerm)) Plugin::unauthorized();
        $this->render();
    }
}
