<?php

namespace CocktailRecipes\Core\Admin;

use CocktailRecipes\Plugin;

/**
 * Trait for building and rendering settings forms
 *
 * Classes using this trait must have a protected $slug property
 */
trait HasForms
{
    // ---------------------------------------------------------------------
    // Internal properties
    // ---------------------------------------------------------------------
    // These properties are used within the class and should not be
    // modified directly

    /** @var string[] text to show before first section */
    protected array $intro = [];

    /** @var array<string,Settings> page forms (formId => Settings) */
    protected array $forms = [];

    /** @var array<string,array> form buttons (formId => button options) */
    protected array $buttons = [];

    /** @var ?string active formId */
    protected ?string $formId = null;

    /** @var bool true if active form has no settings */
    protected bool $noSettings;

    /** @var ?string active sectionId */
    protected ?string $sectionId = null;


    // ---------------------------------------------------------------------
    // Public interfaces
    // ---------------------------------------------------------------------

    /** Add introductory content to display before first section */
    final protected function addIntro(string ...$text): void
    {
        array_push($this->intro, ...$text);
    }

    /**
     * Specifices the settings group to be used
     *
     * Must be called before any sections or inputs are added.
     * Each call creates a new input form on the admin page.
     */
    final protected function addSettings(string $name): void
    {
        $settings = Settings::register($name);
        $this->addForm($settings);
    }

    /** Specifies that no settings group is used on the page/tab */
    final protected function noSettings(): void
    {
        $this->addForm(null);
    }

    /**
     * Set properties for the current form's button
     *
     * Options
     *   warning    string      content to show before submit button
     *   text       string      submit button text; default is "Save Changes"
     *   type       string      button type: primary (default), secondary, delete, small, large
     */
    final protected function setButton(array $options): void
    {
        if (!$this->formId || $this->noSettings) return;
        $this->buttons[$this->formId] = $options;
    }

    /**
     * Add a settings section to an admin page
     *
     * Must be called before any inputs are added.
     * Must call addSettings() before adding any sections
     *
     * @param string $title         text shown as <h2> element at start of input section
     * @param string|array $text    text to show under title to describe the section
     * @param string $id            unique identifier for the text (default is sequentially assigned)
     */
    final protected function addSection(string $title, $text, ?string $id = null): void
    {
        static $counter = 0;
        if (!$this->formId) return;
        add_settings_section(
            $this->sectionId = $id ?? 'page_section_' . ++$counter,
            $title,
            function () use ($text) {
                if (!$text) return;
                foreach (is_array($text) ? $text : [$text] as $p) {
                    echo '<p>' . wp_kses_post($p) . '</p>';
                }
            },
            $this->formId
        );
    }

    /**
     * Add an input field to the current admin page section
     *
     * Options
     *   label      string          shown before input field; title treated as label otherwise
     *   text       string          shown after input field (not for radio/color/textarea inputs)
     *   note       string|array    shown under input field
     *   hint       string          placeholder text for text/numeric input fields
     *   class      string          added by WordPress to <tr> element for the input
     *   style      string          bool: checkbox (default) | radio | select
     *                              enum: radio (default) | select
     *                              num: number (default) | select
     *                              text (font style): regular (default) | code
     *                              textarea (font style): regular | code (default)
     *   values     array           bool: text for radio|select as ['0' => text, '1' => text]
     *                              enum: text for each value as [val => text, ...]
     *   required   bool            input from use is required
     *
     * @param string $title     text shown in left column of setting page
     * @param string $field     input field; also used a id for HTML element
     * @param array  $options   rendering options
     */
    final protected function addInput(string $title, string $field, array $options = []): void
    {
        if (!$this->sectionId || $this->noSettings) return;
        if (isset($options['label'])) unset($options['label_for']);
        $options['_ff'] = $this->formId . ':' . $field;
        add_settings_field($field, $title, [$this, 'renderInput'], $this->formId, $this->sectionId, $options);
    }

    /**
     * Add a group of input fields to the current admin page section
     *
     * Options
     *   id         string      unique identifier for the text (default is sequentially assigned)
     *   class      string      added by WordPress to <tr> element for the input
     *
     * @param string $title     text shown in left column of setting page
     * @param array  $fields    [ field => options, | text ...]
     * @param array  $options   rendering options
     *
     * @see Page::addInput() for details on field options
     */
    final protected function addInputs(string $title, array $fields, array $options = []): void
    {
        static $counter = 0;
        if (!$this->sectionId || $this->noSettings) return;
        $id = $options['id'] ?? 'page_inputs_' . ++$counter;
        add_settings_field($id, $title, [$this, 'renderInputs'], $this->formId, $this->sectionId, [
            'form'   => $this->formId,
            'fields' => $fields,
            'class'  => $options['class'] ?? '',
        ]);
    }

    /**
     * Add fixed text to the current admin page section
     *
     * Options
     *   id         string      unique identifier for the text (default is sequentially assigned)
     *   class      string      added by WordPress to <tr> element for the input
     *
     * @param string $title     text shown in left column of setting page
     * @param string $text      text shown in right column of setting page
     * @param array  $options   rendering options
     */
    final protected function addText(string $title, string $text, array $options = []): void
    {
        static $counter = 0;
        if (!$this->sectionId) return;
        $id = $options['id'] ?? 'page_text_' . ++$counter;
        add_settings_field($id, $title, [$this, 'renderText'], $this->formId, $this->sectionId, [
            'text'  => $text,
            'class' => $options['class'] ?? '',
        ]);
    }

    /**
     * Add an action button to the current section
     *
     * Options
     *   id         string      unique identifier for the button (default is sequentially assigned)
     *   type       string      button type: `primary [large|small]` or `secondary [large|small]`
     *   warning    string      content to show before the button
     *   class      string      used by WordPress to add to <tr> element for the input
     *
     * @param string $action    Action slug (must be defined in Plugin::ADMIN_ACTIONS)
     * @param string $label     Button text
     * @param array  $options   Button options (id, type, class, etc.)
     */
    final protected function addActionButton(string $action, string $label, array $options = []): void
    {
        static $counter = 0;
        if (!$this->sectionId) return;
        $id = $options['id'] ?? 'action_btn_' . ++$counter;
        add_settings_field($id, '', [$this, 'renderActionButton'], $this->formId, $this->sectionId, [
            'slug'  => $action,
            'label' => $label,
            'type'  => $options['type'] ?? 'primary',
            'id'    => $id,
            'warn'  => $options['warning'] ?? '',
            'class' => $options['class'] ?? '',
        ]);
    }

    /**
     * Add a group of action buttons to the current section
     *
     * Row Options
     *   warning    string      content to show before the buttons
     *   class      string      used by WordPress to add to <tr> element for the input
     *
     * Button Options
     *   id         string      unique identifier for the button
     *   label      string      button text
     *   type       string      button type: `primary [large|small]` or `secondary [large|small]`
     *
     * @param array  $buttons       [ actionSlug => label|buttonOptions, ...]
     * @param array  $rowOptions    row rendering options
     */
    final protected function addActionButtons(array $buttons, array $rowOptions = []): void
    {
        static $counter = 0;
        if (!$this->sectionId) return;
        $id = $rowOptions['id'] ?? 'action_btns_' . ++$counter;
        add_settings_field($id, '', [$this, 'renderActionButtons'], $this->formId, $this->sectionId, [
            'btns'  => $buttons,
            'id'    => $id,
            'warn'  => $rowOptions['warning'] ?? '',
            'class' => $rowOptions['class'] ?? '',
        ]);
    }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------

    /** Add a form with optional settings to use */
    private function addForm(?Settings $settings): void
    {
        $formNum = count($this->forms);
        $this->formId = !$formNum ? $this->slug : $this->slug . '_' . ++$formNum;
        $this->forms[$this->formId] = $settings;
        $this->noSettings = $settings === null;
        $this->sectionId = null;
    }

    // render the intro content
    final public function renderIntro(): void
    {
        ?>
        <div class="isgdev-admin-intro">
            <?php foreach ($this->intro as $text): ?>
                <p><?php echo wp_kses_post($text) ?></p>
            <?php endforeach; ?>
        </div>
        <?php
    }

    // render a single input field
    final public function renderInput(array $args, ?string $formId = null, ?string $field = null): void
    {
        if ($formId === null) {
            [$formId, $field] = explode(':', $args['_ff']);
        }
        $settings = $this->forms[$formId];
        $group = $settings::OPTION_NAME;
        $data  = $settings->fieldData($field);

        switch ($data['type']) {

            case 'bool':
                $style = $args['style'] ?? 'checkbox';
                if ($style == 'checkbox') {
                    (new Checkbox($group, $field, $data))->render($args);
                } else {
                    if (!isset($args['values']['0'])) $args['values']['0'] = Plugin::__('off');
                    if (!isset($args['values']['1'])) $args['values']['1'] = Plugin::__('on');
                    ($style == 'radio')
                        ? (new RadioField($group, $field, $data))->render($args)
                        : (new SelectField($group, $field, $data))->render($args);
                }
                break;

            case 'enum':
                $style = $args['style'] ?? 'radio';
                foreach ($data['values'] as $value) {
                    if (!isset($args['values'][$value])) $args['values'][$value] = $value;
                }
                ($style == 'radio')
                    ? (new RadioField($group, $field, $data))->render($args)
                    : (new SelectField($group, $field, $data))->render($args);
                break;

            case 'num':
                if ($values = $data['values'] ?? null) {
                    foreach ($values as $value) {
                        if (!isset($args['values'][$value])) $args['values'][$value] = (string) $value;
                    }
                    ksort($args['values'], SORT_NUMERIC);
                    (new SelectField($group, $field, $data))->render($args);
                } elseif (
                    ($args['style'] ?? null) === 'select'
                    && ($min = $data['min'] ?? null) !== null
                    && ($max = $data['max'] ?? null) !== null
                    && $min <= $max
                ) {
                    $step = $data['step'] ?? 1;
                    for ($value = $min; $value <= $max; $value += $step) {
                        if (!isset($args['values'][$value])) $args['values'][$value] = (string) $value;
                    }
                    ksort($args['values'], SORT_NUMERIC);
                    (new SelectField($group, $field, $data))->render($args);
                } else {
                    (new NumberField($group, $field, $data))->render($args);
                }
                break;

            case 'color':
                (new ColorPicker($group, $field, $data))->render($args);
                break;

            case 'textarea':
            case 'markdown':
                (new Textarea($group, $field, $data))->render($args);
                break;

            default:
                (new TextField($group, $field, $data))->render($args);
        }
    }

    // render fixed text
    final public function renderText(array $args): void
    {
        echo wp_kses_post($args['text']);
    }

    // render a set of input fields
    final public function renderInputs(array $args): void
    {
        $formId = $args['form'];
        $count  = 0;
        echo "<fieldset>";
        foreach ($args['fields'] as $field => $options) {
            if ($count++) echo '<br>';
            if (is_string($options)) {
                echo wp_kses_post($options);
            } else {
                unset($options['label_for']);
                $this->renderInput($options, $formId, $field);
            }
        }
        echo "</fieldset>";
    }

    // render an action button
    final public function renderActionButton(array $args): void
    {
        $this->renderActionFormWarning($ags['warn'] ?? '')
            ->renderActionForm($args['slug'])
            ->renderActionFormButton($args['label'], $args['type'], $args['id'])
            ->renderActionFormEnd();
    }

    // render a row of action buttons
    final public function renderActionButtons(array $args): void
    {
        $this->renderActionFormWarning($ags['warn'] ?? '');
        $counter = 0;
        foreach ($args['btns'] as $slug => $opts) {
            if (!is_array($opts)) $opts = ['label' => $opts];
            $this->renderActionForm($slug, 'isgdev-admin-multiple-btns')
                ->renderActionFormButton(
                    $opts['label'] ?? $slug,
                    $opts['type'] ?? 'primary',
                    $opts['id'] ?? $args['id'] . '_' . ++$counter
                )->renderActionFormEnd();
        }
    }

    private function renderActionFormWarning(string $text = ''): self
    {
        if ($text) {
            echo '<p class="description action-btn-warning">' . wp_kses_post($text) . '</p>';
        }
        return $this;
    }

    private function renderActionForm(string $slug, string $class = ''): self
    {
        static $url;
        $url ??= admin_url('admin-post.php');
        $action = Action::name($slug);
        echo '<form method="post" action="' . esc_url($url) . '" class="' . esc_attr($class) . '">';
        echo '<input type="hidden" name="action" value="' . esc_attr($action) . '">';
        wp_nonce_field($action);
        return $this;
    }

    private function renderActionFormButton(string $label, string $type, string $id): self
    {
        submit_button($label, $type, 'submit', false, ['id' => $id]);
        return $this;
    }

    private function renderActionFormEnd(): self
    {
        echo '</form>';
        return $this;
    }
}
