<?php

namespace CocktailRecipes\Core\Admin;

/**
 * Abstract Admin Page Input Field
 */
abstract class Field
{
    protected const AUTO_RENDER_LABEL = true;
    protected const LABEL_INPUT_WRAPPER = false;
    protected const NOTE_SPACER = true;

    protected string $group;
    protected string $field;
    protected array  $data;
    protected array $options;

    private bool $labelOpened = false;
    private int $brCount = 0;

    final public function __construct(string $group, string $field, array $data)
    {
        $this->group = $group;
        $this->field = $field;
        $this->data  = $data;
    }

    public function render(array $options): void
    {
        $this->options = $options;
        if (static::LABEL_INPUT_WRAPPER) echo '<div class="label-input-wrapper">';
        if (static::AUTO_RENDER_LABEL) $this->renderLabel();
        $this->renderInput($options);
        $this->closeLabel();
        if (static::LABEL_INPUT_WRAPPER) echo '</div>';
        $this->renderNotes();
    }

    protected function renderLabel(): void
    {
        $for = $this->group . '__' . $this->field;
        if ($label = $this->options['label'] ?? '') {
            echo '<label for="' . esc_attr($for) . '">' . esc_html($label) . '</label>';
        } else {
            echo '<label for="' . esc_attr($for) . '">';
            $this->labelOpened = true;
        }
    }

    abstract protected function renderInput(array $options): void;

    protected function closeLabel(): void
    {
        if ($this->labelOpened) echo '</label>';
    }

    protected function renderNotes(): void
    {
        if ($notes = $this->options['note'] ?? null) {
            if (!is_array($notes)) $notes = [$notes];
            if (static::NOTE_SPACER) echo '<br>';
            ?>
                <?php foreach ($notes as $note): ?>
                    <p class="description"><?php echo esc_html($note); ?></p>
                <?php endforeach; ?>
            <?php
        }
    }

    // Helper to output HTML name attribute
    protected function name()
    {
        echo 'name="' . esc_attr($this->group) . '[' . esc_attr($this->field) . ']"';
    }

    // Helper to output HTML id attribute
    protected function id(): void
    {
        echo 'id="' . esc_attr($this->group) . '__' . esc_attr($this->field) . '"';
    }

    // Helper to output HTML value attribute
    protected function value($value = null): void
    {
        echo 'value="' . esc_attr($value ?? $this->data['value']) . '"';
    }

    // Helper to output required attribute
    protected function required(): void
    {
        if ($this->options['required'] ?? false) echo 'required="required"';
    }

    // Helper to output placeholder attribute
    protected function placeholder(): void
    {
        if (($hint = trim($this->options['hint'] ?? '')) != '') {
            echo 'placeholder="' . esc_attr($hint) . '"';
        }
    }

    // Helper to output <br> between elements; skipped on first call
    protected function br(): void
    {
        if ($this->brCount++) echo '<br>';
    }
}
