<?php

namespace CocktailRecipes\Core\Admin;

use CocktailRecipes\Plugin;

/**
 * Abstract Admin Action
 *
 * Base class for admin actions which can be triggered from buttons and
 * controls within the admin pages.
 */
abstract class Action
{
    // ---------------------------------------------------------------------
    // Primary extension points
    // ---------------------------------------------------------------------
    // These are the extension points which an admin action must implement

    /**
     * Handle action
     *
     * @return ?string  optional id of message to be shown in admin
     */
    abstract protected static function execute(): ?string;


    // ---------------------------------------------------------------------
    // Optional extension points
    // ---------------------------------------------------------------------
    // These extension points can be overridden as needed

    /** Admin path, path and/or tab to redirect to after action completes */
    protected const REDIRECT = [];

    /**
     * Required permissions (i.e. WordPress capability) to perform action
     *
     * @see https://wordpress.org/documentation/article/roles-and-capabilities/
     */
    protected const PERMISSIONS = 'manage_options';


    // ---------------------------------------------------------------------
    // Internal properties
    // ---------------------------------------------------------------------
    // These properties are used within the class and should not be
    // modified directly

    /** @var string tab slug used in URL */
    protected string $slug;


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------

    final public static function name(string $actionSlug): string
    {
        return Plugin::slug() . '_' . $actionSlug;
    }

    final public static function handle(string $slug): void
    {
        if (!current_user_can(static::PERMISSIONS)) Plugin::unauthorized();
        check_admin_referer(self::name($slug));
        $message = static::execute();
        if (static::REDIRECT) {
            $url = Page::adminUrl(...static::REDIRECT);
            if ($message) $url = add_query_arg('admin_msg', $message, $url);
        } else {
            $url = Page::adminUrl('index.php');
        }
        wp_safe_redirect($url);
        exit;
    }
}
