<?php

namespace CocktailRecipes\Admin;

use CocktailRecipes\Core\Admin\Tab;
use CocktailRecipes\Core\Helpers\HTML;
use CocktailRecipes\Recipes\Helpers\TokenMap;
use CocktailRecipes\Recipes\Elements\Element;

class ReferenceTab extends Tab
{
    private const BASE_NS = 'CocktailRecipes\Recipes\Elements\\';

    private ?array $map = null;

    public function init(): void
    {
        $this->title = __('Reference', 'cocktail-recipes');
    }

    public function initContent(): void
    {
        $this->noSettings();
        $this->map = TokenMap::get();

        $this->addIntro(
            __('This reference page provides information on creating cocktail recipes using the [cocktail] shortcode, including supported elements, terms and formatting.', 'cocktail-recipes'),
            sprintf(
                /* translators: %s is clickable link */
                __('For complete documentation, visit %s.', 'cocktail-recipes'),
                '<a href="https://www.isgdev.com/software/cocktail-recipes#documentation" target="_blank" rel="noopener noreferrer">'
                    . __('the plugin documentation page', 'cocktail-recipes')
                    . '</a>'
            )
        );

        // Basic Usage
        $this->addSection(
            __('Basic Usage', 'cocktail-recipes'),
            __('Quick start guide for creating cocktail recipes.', 'cocktail-recipes')
        );
        $this->addText(
            _x('Example Recipe', 'sample cocktail recipe code', 'cocktail-recipes'),
            '<pre class="cocktail-recipe-admin-example">'
                . "[cocktail]\n"
                . "2 oz gin\n"
                . "1 oz lemon juice\n"
                . "0.5 oz simple syrup\n"
                . "method: shake\n"
                . "glass: coupe\n"
                . "garnish: lemon twist\n"
                . '[/cocktail]</pre>'
        );
        $this->addText(
            __('How to Specify Elements', 'cocktail-recipes'),
            HTML::markdown(__(
                'Most elements can be entered using **Terms** (like `rocks glass`) with or without a label prefix. **Short Terms** must always be entered with a label (like `ice: yes`). See the reference sections below for all available terms.',
                'cocktail-recipes'
            ))
        );
        $this->addText(
            __('Quantities', 'cocktail-recipes'),
            HTML::markdown(__(
                'Use whole numbers (2), decimals (1.5), fractions (1/2, 2-1/2), or ranges (2-3). Common units: **oz**, **ml**, **cl**, **tsp**, **tbsp**, **dash**, **drop**, **splash**, **bar spoon**',
                'cocktail-recipes'
            ))
        );
        $this->addText(
            __('Modifiers', 'cocktail-recipes'),
            HTML::markdown(__(
                'Add modifiers in square brackets: `[opt]` or `[optional]` for optional ingredients/garnishes, `[alt]` or `[alternate]` for alternative options',
                'cocktail-recipes'
            ))
        );

        // Methods
        $this->addSection(
            __('Mixing Methods', 'cocktail-recipes'),
            HTML::markdown(__(
                'Available mixing methods for cocktail recipes. Labels for methods which can be used include:',
                'cocktail-recipes'
            )) . ' ' . $this->keywordList('method')
        );
        $this->renderElements('method');

        // Glassware
        $this->addSection(
            __('Glassware', 'cocktail-recipes'),
            HTML::markdown(__(
                'Available glassware types for cocktail recipes. Labels for glassware which can be used include:',
                'cocktail-recipes'
            )) . ' ' . $this->keywordList('glass')
        );
        $this->renderElements('glass');

        // Ice
        $this->addSection(
            __('Ice Types', 'cocktail-recipes'),
            HTML::markdown(__(
                'Available ice types for cocktail recipes. Labels for ice types which can be used include:',
                'cocktail-recipes'
            )) . ' ' . $this->keywordList('ice')
        );
        $this->renderElements('ice');

        // Other Elements
        $this->addSection(
            __('Other Elements', 'cocktail-recipes'),
            __('Specifying other recipe elements', 'cocktail-recipes')
        );
        $this->addText(
            __('Ingredients', 'cocktail-recipes'),
            HTML::markdown(__(
                'Ingredients can be entered directly on their own lines or prefixed with:',
                'cocktail-recipes'
            )) . ' ' . $this->keywordList('ingredient')
        );
        $this->addText(
            __('Garnishes', 'cocktail-recipes'),
            HTML::markdown(__(
                'Garnishes must be entered prefixed with these labels:',
                'cocktail-recipes'
            )) . ' ' . $this->keywordList('garnish')
        );
        $this->addText(
            __('Notes', 'cocktail-recipes'),
            HTML::markdown(__(
                'Notes can be added to recipes with these labels:',
                'cocktail-recipes'
            )) . ' ' . $this->noteKeywords('note')
        );
        $this->addText(
            __('Sources', 'cocktail-recipes'),
            HTML::markdown(__(
                'Sources or credits for a recipe can be added with these labels:',
                'cocktail-recipes'
            )) . ' ' . $this->noteKeywords('source')
        );
    }

    // Render the terms, short terms and/or alternatates for all elements of a specific type
    private function renderElements(string $type): void
    {
        foreach ($this->recipeElements($type) as $name) {
            $class   = $this->elementClass($type, $name);
            $element = new $class();
            $content = '<div class="cocktail-recipe-admin-ref-element">';
            // Terms
            if ($terms = $class::terms()) {
                $content .= '<p><strong>'
                    . esc_html_x('Terms', 'recipe element terms (not terminology)', 'cocktail-recipes')
                    . ':</strong> ';
                foreach (explode(',', $terms) as $i => $term) {
                    $content .= ($i ? ', ' : '') . '<code>' . esc_html($term) . '</code>';
                }
                $content .= '</p>';
            }
            // Short Terms
            if ($shortTerms = $class::shortTerms()) {
                $content .= '<p><strong>'
                    . esc_html_x('Short Terms', 'short recipe element terms', 'cocktail-recipes')
                    . ':</strong> ';
                foreach (explode(',', $shortTerms) as $i => $term) {
                    $content .= ($i ? ', ' : '') . '<code>' . esc_html($term) . '</code>';
                }
                $content .= '</p>';
            }
            if (!$terms && !$shortTerms) {
                $content .= esc_html__('No special terms defined', 'cocktail-recipes');
            }
            // Alternates
            if (property_exists($element, 'alt') && $element->alt) {
                $altNames = array_map(function ($name) use ($type) {
                    $altClass = $this->elementClass($type, $name);
                    $altElement = new $altClass();
                    return $altElement->name;
                }, $element->alt);
                $content .= '<p><strong>'
                    . esc_html_x('Alternatives', 'alternative glassware/ice options', 'cocktail-recipes')
                    . ':</strong> ';
                $content .= esc_html(implode(', ', $altNames)) . '</p>';
            }
            $content .= '</div>';
            $this->addText($element->name, $content);
        }
    }

    // Get list of keywords for an element type, ready for including as page content
    private function keywordList($type): string
    {
        $class = $this->elementClass($type);
        $list = '';
        if ($keywords = $class::keywords()) {
            foreach (explode(',', $keywords) as $i => $keyword) {
                $list .= ($i ? ', ' : '') . '<code>' . esc_html($keyword) . ':</code>';
            }
        }
        return $list;
    }

    // Get list of note keywords, ready for including as page content
    private function noteKeywords($noteType): string
    {
        $match = '#' . ucfirst($noteType);
        $list = '';
        $num = 0;
        foreach ($this->map['keywords'] as $keyword => $type) {
            if ($type === $match) {
                $list .= ($num++ ? ', ' : '') . '<code>' . esc_html($keyword) . ':</code>';
            }
        }
        return $list;
    }

    // Get all element names for this type from the token map
    private function recipeElements(string $type): array
    {
        $elements = [];
        // Collect from short terms
        if (isset($this->map[$type])) {
            foreach ($this->map[$type] as $className) {
                $elements[$className] = true;
            }
        }
        // Collect from full terms
        foreach ($this->map['tokens'] as $tokenData) {
            [$tokenType, $className] = explode('.', $tokenData);
            if ($tokenType == $type) {
                $elements[$className] = true;
            }
        }
        ksort($elements);
        return array_keys($elements);
    }

    // Get the full class name for an element
    private function elementClass(string $type, ?string $name = null): string
    {
        $name ??= ucfirst($type);
        return self::BASE_NS . Element::group($type) . '\\' . $name;
    }
}
