<?php

namespace CocktailRecipes\Admin;

use CocktailRecipes\Core\Admin\Tab;
use CocktailRecipes\Core\Helpers\Cache;
use CocktailRecipes\Plugin;
use CocktailRecipes\Recipes\Helpers\TokenMap;
use CocktailRecipes\Shortcodes\CocktailShortcode;

class AdvancedTab extends Tab
{

    public function init(): void
    {
        $this->title = __('Advanced', 'cocktail-recipes');
    }

    public function initContent(): void
    {
        $this->noSettings();

        $this->addIntro(
            __('Advanced settings and diagnostic information for troubleshooting and cache management.', 'cocktail-recipes')
        );

        $this->addSection(__('System Information', 'cocktail-recipes'), '');

        $this->addText(__('Cache', 'cocktail-recipes'), $this->cacheInfo());
        $this->addActionButtons([
            'clear-recipes' => __('Clear Recipe Cache Entries', 'cocktail-recipes'),
            'clear-cache'   => __('Clear All Cache Entries', 'cocktail-recipes')
        ]);

        $this->addText(__('Token Map', 'cocktail-recipes'), $this->tokenMapInfo());
        $this->addActionButton('build-tokenmap', __('Rebuild Token Map', 'cocktail-recipes'));

        $this->addText(__('Diagnostics', 'cocktail-recipes'), $this->diagnosticsInfo());
    }

    // Return cache info text
    private function cacheInfo(): string
    {
        return $this->rows(
            $this->row(
                esc_html__('Enabled', 'cocktail-recipes'),
                $this->yesNo(Cache::enabled())
            ),
            $this->row(
                esc_html__('Writable', 'cocktail-recipes'),
                $this->yesNo(Cache::writable())
            ),
            $this->row(
                esc_html__('Entries', 'cocktail-recipes'),
                count(Cache::entries())
            ),
            $this->row(
                esc_html__('Recipes', 'cocktail-recipes'),
                count(Cache::entries(CocktailShortcode::RECIPE_CACHE_PREFIX))
            )
        );
    }

    // Return Token Map info text
    private function tokenMapInfo(): string
    {
        $map = TokenMap::get(false);
        $exists = $map !== null;
        return $this->rows(
            $this->row(
                esc_html__('Exists', 'cocktail-recipes'),
                $this->yesNo($exists)
            ),
            $this->row(
                esc_html__('Last Built', 'cocktail-recipes'),
                $exists ? ($map['built'] ?? __('Unknown', 'cocktail-recipes')) : '&mdash;'
            )
        );
    }

    // Return diagnostic info text
    private function diagnosticsInfo(): string
    {
        global $wp_version;

        $phpVer = phpversion();
        $phpMin = '7.4';
        $phpOk = (bool) version_compare($phpVer, $phpMin, '>=');
        $phpWarning = $phpOk
            ? ''
            /* translators: %s is php miniumum version required */
            : ' &mdash; <i>' . sprintf(esc_html__('requires %s+', 'cocktail-recipes'), $phpMin);

        $encoding = mb_internal_encoding();
        $encodingOk = strcasecmp($encoding, 'UTF-8') === 0;
        $encodingWarning = $encodingOk
            ? ''
            : ' &mdash; <i>' . sprintf(
                /* translators: %s is encoding format */
                esc_html__('Non-standard encoding detected (%s). UTF-8 is recommended for proper Unicode support in recipes.', 'cocktail-recipes'),
                esc_html($encoding)) . '</i>';

        return $this->rows(
            $this->row(
                esc_html__('Plugin Version', 'cocktail-recipes'),
                esc_html(Plugin::version())
            ),
            $this->row(
                esc_html__('WordPress Version', 'cocktail-recipes'),
                esc_html($wp_version)
            ),
            $this->row(
                __('PHP Version', 'cocktail-recipes'),
                esc_html($phpVer) . ' ' . $this->okay($phpOk) . $phpWarning
            ),
            $this->row(
                __('Character Encoding', 'cocktail-recipes'),
                esc_html($encoding) . ' ' . $this->okay($encodingOk) . $encodingWarning
            )
        );
    }

    // return a row with as `title: info`
    private function row(string $title, string $info): string
    {
        return "<strong>$title:</strong> $info";
    }

    // return all rows, each wrapped in <p>...</p> tags
    private function rows(string ...$rows): string
    {
        return '<p>' . implode('</p><p>', $rows) . '</p>';
    }

    // Return a green checkmark or red x
    private function okay(bool $state): string
    {
        // U+2713 = Check Mark
        // U+2717 = Ballot X
        return $this->colored($state ? "\u{2713}" : "\u{2717}", $state);
    }

    // Return a green "Yes" or red "No"
    private function yesNo(bool $state): string
    {
        static $yes, $no;
        if ($yes === null) {
            $yes = esc_html__('Yes', 'cocktail-recipes');
            $no = esc_html__('No', 'cocktail-recipes');
        }
        return $this->colored($state ? $yes : $no, $state);
    }

    // Return text in either green/red (if color is bool) or a specific color class
    private function colored(string $text, $color): string
    {
        if (is_bool($color)) $color = $color ? 'green' : 'red';
        return '<span class="isgdev-' . esc_attr($color) . '">' . $text . '</span>';
    }
}
