(function () {

    if ("scrollRestoration" in history) history.scrollRestoration = "manual";

    /** Toggle short/full views */
    function toggleCocktailInstructions(btn) {
        const block = btn.closest(".cocktail");
        if (!block) return;
        const inner = block.querySelector(".cocktail-inner");

        const fullLabel = btn.dataset.fullLabel;
        const shortLabel = btn.dataset.shortLabel;

        const expanding = block.classList.contains("recipe-short");
        const fromState = expanding ? "short" : "full";
        const toState = expanding ? "full" : "short";
        const transitionState = expanding ? "expanding" : "collapsing";

        // Clear any inline height/opacity left from a previous animation
        inner.style.height = "";
        inner.style.opacity = "";

        // Capture current height
        const startHeight = inner.scrollHeight;

        // Switch temporarily to target resting state to measure end height
        block.classList.remove(`recipe-${fromState}`);
        block.classList.add(`recipe-${toState}`);
        const endHeight = inner.scrollHeight;

        // Revert to starting state
        block.classList.remove(`recipe-${toState}`);
        block.classList.add(`recipe-${fromState}`);

        // Apply transition state
        block.classList.add(`recipe-${transitionState}`);
        inner.style.height = startHeight + "px";
        inner.offsetHeight; // reflow

        // Animate to target height
        inner.style.height = endHeight + "px";
        inner.style.opacity = expanding ? 0 : 1;

        inner.addEventListener("transitionend", () => {
            if (transitionState === "collapsing") {
                // Fade out content first
                inner.style.transition = "opacity 0.4s ease";
                inner.style.opacity = 0;
                inner.addEventListener("transitionend", () => {
                    finalize();
                }, { once: true });
            } else {
                finalize();
            }
            function finalize() {
                block.classList.remove(`recipe-${transitionState}`);
                block.classList.remove(`recipe-${fromState}`);
                block.classList.add(`recipe-${toState}`);
                inner.style.height = "";
                inner.style.opacity = "";
                inner.style.transition = "";
            }
        }, { once: true });

        btn.textContent = expanding ? shortLabel : fullLabel;
        return toState;
    }

    let cachedDefaults = null;

    function getDefault(name) {
        if (!cachedDefaults) {
            const el = document.querySelector(".cocktail-control-defaults");
            cachedDefaults = !el ? {} : {
                view: el.dataset.defaultView || '',
                servings: el.dataset.defaultServings || '',
                units: el.dataset.defaultUnits || ''
            };
        }
        return cachedDefaults[name] || '';
    }

    /** Update cocktail view parameters and reload page */
    function updateCocktailParam(btn, key, value, reload) {
        const params = new URLSearchParams(window.location.search);
        const block = btn.closest(".cocktail");
        if (!block) return;
        const suffix = block.dataset.paramSfx || "";
        if (value === getDefault(key)) {
            params.delete(key + suffix);
        } else {
            params.set(key + suffix, value);
        }
        if (reload) {
            sessionStorage.setItem("cocktailScrollPos", window.scrollY);
            window.location.search = params.toString() + window.location.hash;
        } else {
            const qs  = params.toString();
            const url = window.location.pathname + (qs ? "?" + qs : "") + window.location.hash;
            window.history.replaceState({}, "", url);
        }
    }

    /** Bind event listeners for controls */
    document.addEventListener("click", (e) => {
        const target = e.target.closest("[data-cocktail-action]");
        if (!target) return;

        const action = target.dataset.cocktailAction;

        if (action == 'link') {
            // Don't navigate if clicking on a link or button inside the card
            if (e.target.closest("a, button")) return;
            window.location.href = target.dataset.cardLink;
            return;
        }

        e.preventDefault();
        let value, reload;
        if (action == 'view') {
            value = toggleCocktailInstructions(target);
            reload = false;
        } else {
            value = target.dataset.value;
            reload = true;
            const group = target.closest('.cocktail-btn-group');
            if (group) {
                group.querySelectorAll('.cocktail-btn').forEach(btn => {
                    btn.classList.remove('selected');
                    btn.setAttribute('aria-pressed', 'false');
                });
                target.classList.add('selected');
                target.setAttribute('aria-pressed', 'true');
            }
        }
        updateCocktailParam(target, action, value, reload);
    });
    document.addEventListener("keydown", (e) => {
        if (e.key === "Enter" || e.key === " ") {
            const target = e.target.closest("[data-cocktail-action]");
            if (!target || target.tagName !== 'BUTTON') return;
            e.preventDefault();
            target.click();
        }
    });
    document.addEventListener("DOMContentLoaded", () => {
        const pos = sessionStorage.getItem("cocktailScrollPos");
        if (pos !== null) {
            window.scrollTo(0, parseInt(pos, 10));
            sessionStorage.removeItem("cocktailScrollPos");
        }
    });

})();

