<?php
/**
 * Submit REST Controller
 *
 * Purpose: Handle frontend form submissions
 * Location: /rest/frontend/class-submit-controller.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_REST_Submit {

    /**
     * Register REST API routes.
     */
    public static function register() {
        // POST /wp-json/cmbsqf/v1/frontend/submit
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/frontend/submit',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'submit_form'],
                'permission_callback' => '__return_true', // Public endpoint
                'args'                => [
                    'form_id' => [
                        'required'          => true,
                        'type'              => 'integer',
                        'sanitize_callback' => 'absint',
                    ],
                ],
            ]
        );
    }

    /**
     * POST /wp-json/cmbsqf/v1/frontend/submit
     *
     * Handle form submission.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error Response or error.
     */
    public static function submit_form($request) {
        $form_id = $request->get_param('form_id');

        // Validate form_id
        if (!$form_id || $form_id <= 0) {
            return new WP_Error(
                'invalid_form_id',
                __('Invalid form ID.', 'cmb-sqlite-form'),
                ['status' => 400]
            );
        }

        // Rate limiting: configurable submissions per minute per IP
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        $rate_limit_max = (int) CMBSQF_DB_Settings::get('form__bot_protection__rate_limit', $form_id, 5);
        $rate_limit_window = 60; // 60 seconds window
        
        $client_ip = self::get_client_ip();
        $rate_limit_key = 'cmbsqf_submit_' . md5($client_ip . '_' . $form_id);
        $attempts = get_transient($rate_limit_key);

        if ($attempts && $attempts >= $rate_limit_max) {
            return new WP_Error(
                'rate_limit_exceeded',
                sprintf(
                    /* translators: %d: number of seconds to wait before trying again */
                    __('Too many submissions. Please try again in %d seconds.', 'cmb-sqlite-form'),
                    $rate_limit_window
                ),
                ['status' => 429]
            );
        }

        // Increment rate limit counter
        set_transient($rate_limit_key, ($attempts ? $attempts + 1 : 1), $rate_limit_window);

        // Load settings to get honeypot field name
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        $honeypot_field = CMBSQF_DB_Settings::get('form__bot_protection__honeypot_field', $form_id, 'business_url');

        // Collect submission data
        $data = [
            'name'         => sanitize_text_field($request->get_param('name')),
            'surname'      => sanitize_text_field($request->get_param('surname')),
            'email'        => sanitize_email($request->get_param('email')),
            'phone'        => sanitize_text_field($request->get_param('phone')),
            'organization' => sanitize_text_field($request->get_param('organization')),
            'website'      => esc_url_raw($request->get_param('website')),
            'position'     => sanitize_text_field($request->get_param('position')),
            'subject'      => sanitize_text_field($request->get_param('subject')),
            'message'      => sanitize_textarea_field($request->get_param('message')),
            'privacy_status' => sanitize_text_field($request->get_param('privacy_status')),
            // Bot protection fields
            $honeypot_field    => sanitize_text_field($request->get_param($honeypot_field)),
            'form_loaded_at'   => absint($request->get_param('form_loaded_at')),
            'captcha_response' => sanitize_text_field($request->get_param('captcha_response')),
        ];

        // Load dependencies
        require_once CMBSQF_PLUGIN_DIR . 'frontend/class-form-validator.php';
        require_once CMBSQF_PLUGIN_DIR . 'frontend/class-form-handler.php';

        // Validate submission
        $validation_errors = CMBSQF_Form_Validator::validate($form_id, $data);

        if (!empty($validation_errors)) {
            return rest_ensure_response([
                'success' => false,
                'data'    => [
                    'message' => implode(' ', $validation_errors),
                ],
            ]);
        }

        // Process submission
        $result = CMBSQF_Form_Handler::process($form_id, $data);

        if (is_wp_error($result)) {
            return $result;
        }

        // Determine message based on email delivery status
        if (isset($result['email_sent']) && $result['email_sent']) {
            $message = __('Thank you! Your message has been sent successfully.', 'cmb-sqlite-form');
        } else {
            $message = __('Your message has been received, but we couldn\'t send you a confirmation email. We\'ll contact you soon.', 'cmb-sqlite-form');
        }

        return rest_ensure_response([
            'success' => true,
            'data'    => [
                'message' => $message,
            ],
        ]);
    }

    /**
     * Get client IP address for rate limiting.
     * 
     * SECURITY: Uses REMOTE_ADDR exclusively to prevent IP spoofing.
     * 
     * Headers like X-Forwarded-For, HTTP_CLIENT_IP can be trivially 
     * spoofed by attackers to bypass rate limiting. Since this is a 
     * public plugin for unknown environments, we MUST use REMOTE_ADDR 
     * which reflects the actual TCP connection IP.
     * 
     * NOTE: If you're behind a reverse proxy (Cloudflare, ALB, etc.), 
     * REMOTE_ADDR will be the proxy IP. This is acceptable trade-off 
     * for security in a public plugin. Users behind the same proxy 
     * will share the rate limit, which is still better than allowing 
     * unlimited spoofing attacks.
     *
     * @return string Client IP address.
     */
    private static function get_client_ip() {
        // Use REMOTE_ADDR only - cannot be spoofed by client
        if (isset($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
        
        return '0.0.0.0';
    }
}
