<?php
/**
 * Settings REST Controller
 *
 * Purpose: Handle global and form-specific settings
 * Location: /rest/admin/class-settings-controller.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_REST_Settings {

    /**
     * Sanitization rules for setting keys.
     *
     * Pattern types:
     * - 'exact'   : Key must match exactly
     * - 'suffix'  : Key must end with pattern
     * - 'contains': Key must contain pattern
     */
    private const SANITIZATION_RULES = [
        // Exact matches
        'form__styling__custom_css' => ['type' => 'exact', 'sanitizer' => 'wp_strip_all_tags'],

        // Suffix matches (keys ending with)
        '_emails' => ['type' => 'suffix', 'sanitizer' => 'sanitize_email'],
        '_email'  => ['type' => 'suffix', 'sanitizer' => 'sanitize_email'],
        '_url'    => ['type' => 'suffix', 'sanitizer' => 'esc_url_raw'],

        // Contains matches
        'html'     => ['type' => 'contains', 'sanitizer' => 'wp_kses_post'],
        'header'   => ['type' => 'contains', 'sanitizer' => 'wp_kses_post'],
        'message'  => ['type' => 'contains', 'sanitizer' => 'wp_kses_post'],
        'url'      => ['type' => 'contains', 'sanitizer' => 'esc_url_raw'],
        'website'  => ['type' => 'contains', 'sanitizer' => 'esc_url_raw'],
        'enabled'  => ['type' => 'contains', 'sanitizer' => 'boolean'],
        'override' => ['type' => 'contains', 'sanitizer' => 'boolean'],
        'block'    => ['type' => 'contains', 'sanitizer' => 'boolean'],
        'required' => ['type' => 'contains', 'sanitizer' => 'boolean'],
    ];

    /**
     * Register REST API routes.
     */
    public static function register() {
        // POST /wp-json/cmbsqf/v1/admin/settings
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/settings',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'save_settings'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'context' => [
                        'required'          => true,
                        'type'              => 'string',
                        'enum'              => ['global', 'form'],
                        'sanitize_callback' => 'sanitize_key',
                    ],
                    'form_id' => [
                        'required'          => false,
                        'type'              => 'integer',
                        'sanitize_callback' => 'absint',
                    ],
                    'settings' => [
                        'required'          => true,
                        'type'              => 'object',
                        'validate_callback' => function($settings) {
                            return is_array($settings);
                        },
                    ],
                ],
            ]
        );
    }

    /**
     * POST /wp-json/cmbsqf/v1/admin/settings
     *
     * Save settings to database.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error Response or error.
     */
    public static function save_settings($request) {
        $context  = $request->get_param('context');
        $form_id  = $request->get_param('form_id');
        $settings = $request->get_param('settings');

        // Validate form_id if context is 'form'
        if ($context === 'form' && (!$form_id || $form_id <= 0)) {
            return new WP_Error(
                'missing_form_id',
                __('Form ID is required for form context.', 'cmb-sqlite-form'),
                ['status' => 400]
            );
        }

        // Default form_id to 0 for global context
        if ($context === 'global') {
            $form_id = 0;
        }

        // Sanitize and save each setting
        $saved_count = 0;
        $errors      = [];

        // Load CSS Storage helper
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-css-storage.php';

        foreach ($settings as $key => $value) {
            // Remove [] suffix from array fields (e.g., form__sources[] -> form__sources)
            $clean_key = substr($key, -2) === '[]' ? substr($key, 0, -2) : $key;
            $sanitized_key = sanitize_key($clean_key);

            // Special handling for Custom CSS - Save to file, NOT database (both global and form)
            if ($sanitized_key === 'form__styling__custom_css') {
                $css_content = is_string($value) ? $value : '';
                
                $file_result = CMBSQF_CSS_Storage::save($css_content, $form_id);
                
                if ($file_result && !is_wp_error($file_result)) {
                    $saved_count++;
                } else {
                    $errors[] = $key . ' (File Write Error)';
                }
                
                // Skip DB saving for this key
                continue;
            }

            // Special handling for Template Selection - Load template CSS and save to file
            if ($sanitized_key === 'form__styling__template') {
                $template_key = sanitize_key($value);
                $template_file = CMBSQF_PLUGIN_DIR . 'admin/css-templates/' . $template_key . '.css';

                // Validate template path with realpath() to prevent directory traversal
                $real_path = realpath($template_file);
                $templates_dir = realpath(CMBSQF_PLUGIN_DIR . 'admin/css-templates');

                if (!$real_path || strpos($real_path, $templates_dir) !== 0) {
                    $errors[] = $key . ' (Invalid Template)';
                    continue;
                }

                $template_css = file_get_contents($real_path);

                if ($template_css !== false) {
                    $file_result = CMBSQF_CSS_Storage::save($template_css, $form_id);

                    if ($file_result && !is_wp_error($file_result)) {
                        // Also save the template name to DB for reference
                        if ($context === 'global') {
                            CMBSQF_DB_Settings::set_global_setting($sanitized_key, $template_key);
                        } else {
                            CMBSQF_DB_Settings::set_form_setting($sanitized_key, $template_key, $form_id);
                        }
                        $saved_count++;
                    } else {
                        $errors[] = $key . ' (File Write Error)';
                    }
                }

                // Skip normal DB saving for this key (already saved above)
                continue;
            }

            $sanitized_value = self::sanitize_setting_value($clean_key, $value);

            // Save based on context
            if ($context === 'global') {
                $result = CMBSQF_DB_Settings::set_global_setting($sanitized_key, $sanitized_value);
            } else {
                $result = CMBSQF_DB_Settings::set_form_setting($sanitized_key, $sanitized_value, $form_id);
            }

            if ($result) {
                $saved_count++;
            } else {
                $errors[] = $key;
            }
        }

        // Check if any settings were saved
        if ($saved_count === 0 && !empty($settings)) {
            return new WP_Error(
                'save_failed',
                __('Failed to save settings.', 'cmb-sqlite-form'),
                ['status' => 500]
            );
        }

        return rest_ensure_response([
            'success' => true,
            'message' => sprintf(
                // translators: %d is the number of settings saved
                __('%d settings saved successfully.', 'cmb-sqlite-form'),
                $saved_count
            ),
            'saved'   => $saved_count,
            'errors'  => $errors,
        ]);
    }

    /**
     * Sanitize setting value based on key using SANITIZATION_RULES.
     *
     * @param string $key   Setting key.
     * @param mixed  $value Setting value.
     * @return string Sanitized value.
     */
    private static function sanitize_setting_value($key, $value) {
        // Handle arrays (e.g., form__junk_email__sources[])
        if (is_array($value)) {
            $sanitized_array = array_map(function($item) use ($key) {
                if (strpos($key, 'url') !== false || strpos($key, 'source') !== false) {
                    return esc_url_raw($item);
                }
                return sanitize_text_field($item);
            }, $value);

            return wp_json_encode($sanitized_array);
        }

        // Find matching sanitization rule
        $sanitizer = self::find_sanitizer_for_key($key);

        return self::apply_sanitizer($sanitizer, $value);
    }

    /**
     * Find the appropriate sanitizer for a given key.
     *
     * @param string $key Setting key.
     * @return string Sanitizer name.
     */
    private static function find_sanitizer_for_key($key) {
        foreach (self::SANITIZATION_RULES as $pattern => $rule) {
            $matches = false;

            switch ($rule['type']) {
                case 'exact':
                    $matches = ($key === $pattern);
                    break;
                case 'suffix':
                    $matches = (substr($key, -strlen($pattern)) === $pattern);
                    break;
                case 'contains':
                    $matches = (strpos($key, $pattern) !== false);
                    break;
            }

            if ($matches) {
                return $rule['sanitizer'];
            }
        }

        // Default sanitizer
        return 'sanitize_text_field';
    }

    /**
     * Apply sanitizer to value.
     *
     * @param string $sanitizer Sanitizer name.
     * @param mixed  $value     Value to sanitize.
     * @return string Sanitized value.
     */
    private static function apply_sanitizer($sanitizer, $value) {
        switch ($sanitizer) {
            case 'sanitize_email':
                return sanitize_email($value);
            case 'wp_kses_post':
                return wp_kses_post($value);
            case 'esc_url_raw':
                return esc_url_raw($value);
            case 'wp_strip_all_tags':
                return wp_strip_all_tags($value);
            case 'boolean':
                return $value === '1' ? '1' : '0';
            case 'sanitize_text_field':
            default:
                return sanitize_text_field($value);
        }
    }
}
