<?php
/**
 * Forms REST Controller
 *
 * Purpose: Handle form management (create, rename, delete)
 * Location: /rest/admin/class-forms-controller.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_REST_Forms {

    /**
     * Register REST API routes.
     */
    public static function register() {
        // POST /wp-json/cmbsqf/v1/admin/forms (Create Form)
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/forms',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'create_form'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'name' => [
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ]
        );

        // PATCH /wp-json/cmbsqf/v1/admin/forms/{id} (Rename Form)
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/forms/(?P<id>\d+)',
            [
                'methods'             => 'PATCH',
                'callback'            => [__CLASS__, 'rename_form'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'id' => [
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                    'name' => [
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ]
        );

        // DELETE /wp-json/cmbsqf/v1/admin/forms/{id} (Delete Form)
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/forms/(?P<id>\d+)',
            [
                'methods'             => 'DELETE',
                'callback'            => [__CLASS__, 'delete_form'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'id' => [
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                ],
            ]
        );

        // POST /wp-json/cmbsqf/v1/admin/forms/batch (Batch Actions)
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/forms/batch',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'batch_process'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'action' => [
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_key',
                    ],
                    'ids' => [
                        'required'          => true,
                        'type'              => 'array',
                        'validate_callback' => function($param) {
                            return is_array($param) && !empty($param);
                        }
                    ],
                ],
            ]
        );


        // POST /wp-json/cmbsqf/v1/admin/forms/{id}/clear-entries (Clear All Entries)
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/forms/(?P<id>\d+)/clear-entries',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'clear_all_entries'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'id' => [
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                ],
            ]
        );


    }

    /**
     * Create a new form.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public static function create_form($request) {
        $name = sanitize_text_field($request->get_param('name'));

        if (empty($name)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Form name cannot be empty.', 'cmb-sqlite-form'),
            ], 400);
        }

        // Load DB Core
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';

        try {
            $form_id = CMBSQF_DB_Core::create_form($name);

            if (is_wp_error($form_id)) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => $form_id->get_error_message(),
                ], 500);
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => __('Form created successfully.', 'cmb-sqlite-form'),
                'data'    => [
                    'id'   => $form_id,
                    'name' => $name,
                ],
            ], 201);
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Rename a form.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public static function rename_form($request) {
        $form_id = (int) $request->get_param('id');
        $name    = sanitize_text_field($request->get_param('name'));

        if (empty($name)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Form name cannot be empty.', 'cmb-sqlite-form'),
            ], 400);
        }

        // Load DB Core
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';

        try {
            $success = CMBSQF_DB_Core::rename_form($form_id, $name);

            if ($success) {
                return new WP_REST_Response([
                    'success' => true,
                    'message' => __('Form renamed successfully.', 'cmb-sqlite-form'),
                    'data'    => [
                        'id'   => $form_id,
                        'name' => $name,
                    ],
                ], 200);
            } else {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('Failed to rename form.', 'cmb-sqlite-form'),
                ], 500);
            }
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete a form.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public static function delete_form($request) {
        $form_id = (int) $request->get_param('id');

        // Load DB Core
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';

        try {
            $success = CMBSQF_DB_Core::delete_form($form_id);

            if ($success) {
                return new WP_REST_Response([
                    'success' => true,
                    'message' => __('Form deleted successfully.', 'cmb-sqlite-form'),
                    'data'    => [
                        'id' => $form_id,
                    ],
                ], 200);
            } else {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('Failed to delete form.', 'cmb-sqlite-form'),
                ], 500);
            }
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Process batch actions.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public static function batch_process($request) {
        $action = $request->get_param('action');
        $ids    = array_map('absint', (array) $request->get_param('ids'));

        if ($action !== 'delete') {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Invalid batch action.', 'cmb-sqlite-form'),
            ], 400);
        }

        if (empty($ids)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('No forms selected.', 'cmb-sqlite-form'),
            ], 400);
        }

        // Load DB Core
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';

        $deleted_count = 0;

        foreach ($ids as $id) {
            if ($id <= 0) continue;

            if (CMBSQF_DB_Core::delete_form($id)) {
                $deleted_count++;
            }
        }

        if ($deleted_count > 0) {
            return new WP_REST_Response([
                'success' => true,
                'message' => sprintf(
                    /* translators: %d: number of forms deleted */
                    _n('%d form deleted successfully.', '%d forms deleted successfully.', $deleted_count, 'cmb-sqlite-form'),
                    $deleted_count
                ),
            ], 200);
        } else {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Failed to delete forms.', 'cmb-sqlite-form'),
            ], 500);
        }
    }


    /**
     * Clear all entries for a form.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public static function clear_all_entries($request) {
        $form_id = (int) $request->get_param('id');

        // Load DB Submissions
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';

        try {
            $deleted_count = CMBSQF_DB_Submissions::delete_all_by_form($form_id);

            if ($deleted_count === false) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('Failed to clear entries.', 'cmb-sqlite-form'),
                ], 500);
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => sprintf(
                    /* translators: %d: number of entries deleted */
                    _n('%d entry deleted successfully.', '%d entries deleted successfully.', $deleted_count, 'cmb-sqlite-form'),
                    $deleted_count
                ),
                'data' => [
                    'deleted_count' => $deleted_count,
                ],
            ], 200);

        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

}
