<?php
/**
 * REST API Controller for Entries
 * 
 * Purpose: Handle entry CRUD operations via REST API
 * Location: /rest/admin/class-entries-controller.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_REST_Entries_Controller extends WP_REST_Controller {
    
    /**
     * Register REST routes
     */
    public function register_routes() {
        // GET /entries/{id} - Get single entry
        register_rest_route(CMBSQF_Constants::REST_NAMESPACE, '/admin/entries/(?P<id>\\d+)', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_item'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'id' => [
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                    'form_id' => [
                        'required' => true,
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                ],
            ],
        ]);
        
        // PATCH /entries/{id} - Update entry fields
        register_rest_route(CMBSQF_Constants::REST_NAMESPACE, '/admin/entries/(?P<id>\\d+)', [
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [$this, 'update_item'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'id' => [
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                    'form_id' => [
                        'required' => true,
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                    'name' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'surname' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'email' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_email',
                    ],
                    'phone' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'organization' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'position' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'website' => [
                        'type' => 'string',
                        'sanitize_callback' => 'esc_url_raw',
                    ],
                    'subject' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'message' => [
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_textarea_field',
                    ],
                    'privacy_status' => [
                        'type' => 'string',
                        'enum' => ['not_configured', 'not_accepted', 'accepted'],
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);
        
        // POST /entries/{id}/resend - Resend autoresponder
        register_rest_route(CMBSQF_Constants::REST_NAMESPACE, '/admin/entries/(?P<id>\\d+)/resend', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'resend_email'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'id' => [
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                    'form_id' => [
                        'required' => true,
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                ],
            ],
        ]);
        
        // DELETE /entries/{id} - Delete entry
        register_rest_route(CMBSQF_Constants::REST_NAMESPACE, '/admin/entries/(?P<id>\\d+)', [
            [
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => [$this, 'delete_item'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'id' => [
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                    'form_id' => [
                        'required' => true,
                        'validate_callback' => function($param) {
                            return is_numeric($param);
                        }
                    ],
                ],
            ],
        ]);
    }
    
    /**
     * Get single entry by ID
     */
    public function get_item($request) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        
        $entry_id = (int) $request['id'];
        $form_id = (int) $request['form_id'];
        
        try {
            $entry = CMBSQF_DB_Submissions::get_by_id($entry_id, $form_id);
            
            if (!$entry) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Entry not found',
                ], 404);
            }
            
            return new WP_REST_Response([
                'success' => true,
                'data' => $entry,
            ], 200);
            
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
    
    /**
     * Update entry fields
     */
    public function update_item($request) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        
        $entry_id = (int) $request['id'];
        
        // Build update data from all allowed fields
        $update_data = [];
        
        // Text fields
        $text_fields = ['name', 'surname', 'phone', 'organization', 'position', 'subject'];
        foreach ($text_fields as $field) {
            if (isset($request[$field])) {
                $update_data[$field] = sanitize_text_field($request[$field]);
            }
        }
        
        // Email field (with validation)
        if (isset($request['email'])) {
            $email = sanitize_email($request['email']);
            if (!empty($email) && !is_email($email)) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Invalid email format',
                ], 400);
            }
            $update_data['email'] = $email;
        }
        
        // Website field (with URL validation)
        if (isset($request['website'])) {
            $website = esc_url_raw($request['website']);
            $update_data['website'] = $website;
        }
        
        // Message field (textarea)
        if (isset($request['message'])) {
            $update_data['message'] = sanitize_textarea_field($request['message']);
        }
        
        // Privacy status field (with enum validation)
        if (isset($request['privacy_status'])) {
            $privacy_status = sanitize_text_field($request['privacy_status']);
            $allowed_statuses = ['not_configured', 'not_accepted', 'accepted'];
            
            if (!in_array($privacy_status, $allowed_statuses, true)) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Invalid privacy status value',
                ], 400);
            }
            
            $update_data['privacy_status'] = $privacy_status;
        }
        
        if (empty($update_data)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'No valid fields to update',
            ], 400);
        }
        
        try {
            // Get form_id from request or derive from entry
            $form_id = (int) $request->get_param('form_id');
            if (!$form_id) {
                // Try to get from entry first
                $entry = CMBSQF_DB_Submissions::get_by_id($entry_id, 0); // Will fail, need form_id
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Missing form_id parameter',
                ], 400);
            }
            
            $result = CMBSQF_DB_Submissions::update($entry_id, $form_id, $update_data);
            
            if ($result) {
                return new WP_REST_Response([
                    'success' => true,
                    'message' => 'Entry updated successfully',
                ], 200);
            } else {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Failed to update entry',
                ], 500);
            }
            
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
    
    /**
     * Resend autoresponder email
     */
    public function resend_email($request) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-smtp-mailer.php';
        
        $entry_id = (int) $request['id'];
        
        try {
            // Get entry data - now requires form_id from request
            $form_id = (int) $request->get_param('form_id');
            if (!$form_id) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Missing form_id parameter',
                ], 400);
            }
            
            $entry = CMBSQF_DB_Submissions::get_by_id($entry_id, $form_id);
            
            if (!$entry) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Entry not found',
                ], 404);
            }
            
            $form_id = $entry['form_id'];
            
            if (empty($entry['email'])) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Entry has no email address',
                ], 400);
            }
            
            // Get autoresponder settings with simple defaults (no placeholder replacement for resend)
            $subject = CMBSQF_DB_Settings::get('notifications__autoresponder__subject', $form_id, 'Thank you for your message');
            $body = CMBSQF_DB_Settings::get('notifications__autoresponder__body', $form_id, 'We have received your message and will get back to you soon.');
            
            // Load email configuration
            require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
            $config = [
                'smtp_enabled'    => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_enabled', $form_id, '0'),
                'smtp_host'       => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_host', $form_id, ''),
                'smtp_port'       => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_port', $form_id, '587'),
                'smtp_username'   => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_username', $form_id, ''),
                'smtp_password'   => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_password', $form_id, ''),
                'smtp_encryption' => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_encryption', $form_id, 'tls'),
                'from_name'       => CMBSQF_DB_Settings::get('notifications__sender__from_name', $form_id, get_bloginfo('name')),
                'from_email'      => CMBSQF_DB_Settings::get('notifications__sender__from_email', $form_id, get_bloginfo('admin_email')),
            ];
            
            // Send email with debug to capture errors
            $result = CMBSQF_SMTP_Mailer::send_test_email($config, $entry['email'], $subject, $body);
            
            if ($result['success']) {
                // Update delivery status
                CMBSQF_DB_Submissions::update($entry_id, $form_id, [
                    'email_delivery' => 'sent',
                    'email_delivery_error' => null,
                ]);
                
                return new WP_REST_Response([
                    'success' => true,
                    'message' => 'Email sent successfully',
                ], 200);
            } else {
                // Capture detailed error
                $error_msg = 'Failed to send email';
                if (isset($result['error_details']['message'])) {
                    $error_msg = $result['error_details']['message'];
                }
                
                // Update delivery status with error
                CMBSQF_DB_Submissions::update($entry_id, $form_id, [
                    'email_delivery' => 'failed',
                    'email_delivery_error' => $error_msg,
                ]);
                
                return new WP_REST_Response([
                    'success' => false,
                    'message' => $error_msg,
                ], 500);
            }
            
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
    
    /**
     * Delete entry
     */
    public function delete_item($request) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        
        $entry_id = (int) $request['id'];
        $form_id = (int) $request['form_id'];
        
        try {
            $result = CMBSQF_DB_Submissions::delete($entry_id, $form_id);
            
            if ($result) {
                return new WP_REST_Response([
                    'success' => true,
                    'message' => 'Entry deleted successfully',
                ], 200);
            } else {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Failed to delete entry or entry not found',
                ], 404);
            }
            
        } catch (Exception $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
    
    /**
     * Check permissions
     */
    public function check_permission() {
        return current_user_can('manage_options');
    }
}
