<?php
/**
 * Email REST Controller
 *
 * Purpose: Handle email testing and debugging
 * Location: /rest/admin/class-email-controller.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_REST_Email {

    /**
     * Register REST API routes.
     */
    public static function register() {
        // POST /wp-json/cmbsqf/v1/admin/test-email
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/test-email',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'send_test_email'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'recipient' => [
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_email',
                    ],
                    'smtp_enabled' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'smtp_host' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'smtp_port' => [
                        'required'          => false,
                        'type'              => 'integer',
                        'sanitize_callback' => 'absint',
                    ],
                    'smtp_username' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'smtp_password' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'smtp_encryption' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'from_name' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'from_email' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_email',
                    ],
                ],
            ]
        );
    }

    /**
     * POST /wp-json/cmbsqf/v1/admin/test-email
     *
     * Send test email to verify SMTP configuration with detailed debugging.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error Response or error.
     */
    public static function send_test_email($request) {
        $recipient = $request->get_param('recipient');

        // Validate email
        if (!is_email($recipient)) {
            return new WP_Error(
                'invalid_email',
                __('Invalid email address.', 'cmb-sqlite-form'),
                ['status' => 400]
            );
        }

        // Get SMTP settings from request
        $config = [
            'smtp_enabled'    => $request->get_param('smtp_enabled') === '1' ? '1' : '0',
            'smtp_host'       => $request->get_param('smtp_host'),
            'smtp_port'       => $request->get_param('smtp_port'),
            'smtp_username'   => $request->get_param('smtp_username'),
            'smtp_password'   => $request->get_param('smtp_password'),
            'smtp_encryption' => $request->get_param('smtp_encryption'),
            'from_name'       => $request->get_param('from_name'),
            'from_email'      => $request->get_param('from_email'),
        ];

        // Prepare email
        $subject = sprintf(
            /* translators: %s: site/blog name */
            __('Test Email from %s', 'cmb-sqlite-form'),
            get_bloginfo('name')
        );
        $message = '<html><body>';
        $message .= '<h2>' . __('Test Email', 'cmb-sqlite-form') . '</h2>';
        $message .= '<p>' . __('This is a test email to verify your email configuration is working correctly.', 'cmb-sqlite-form') . '</p>';
        $message .= '<hr>';
        $message .= '<h3>' . __('Configuration Details:', 'cmb-sqlite-form') . '</h3>';
        $message .= '<ul>';
        $message .= '<li><strong>' . __('Delivery Method:', 'cmb-sqlite-form') . '</strong> ' . ($config['smtp_enabled'] === '1' ? __('Custom SMTP', 'cmb-sqlite-form') : __('WordPress default (wp_mail)', 'cmb-sqlite-form')) . '</li>';

        if ($config['smtp_enabled'] === '1') {
            $message .= '<li><strong>' . __('SMTP Host:', 'cmb-sqlite-form') . '</strong> ' . esc_html($config['smtp_host']) . '</li>';
            $message .= '<li><strong>' . __('SMTP Port:', 'cmb-sqlite-form') . '</strong> ' . esc_html($config['smtp_port']) . '</li>';
            $message .= '<li><strong>' . __('SMTP Encryption:', 'cmb-sqlite-form') . '</strong> ' . esc_html(strtoupper($config['smtp_encryption'])) . '</li>';
            $message .= '<li><strong>' . __('SMTP Authentication:', 'cmb-sqlite-form') . '</strong> ' . (!empty($config['smtp_username']) ? __('Enabled', 'cmb-sqlite-form') : __('Disabled', 'cmb-sqlite-form')) . '</li>';
        }

        $message .= '<li><strong>' . __('From Name:', 'cmb-sqlite-form') . '</strong> ' . esc_html($config['from_name']) . '</li>';
        $message .= '<li><strong>' . __('From Email:', 'cmb-sqlite-form') . '</strong> ' . esc_html($config['from_email']) . '</li>';
        $message .= '</ul>';
        $message .= '<hr>';
        $message .= '<p><small>' . sprintf(
            /* translators: 1: date format, 2: time format */
            __('Sent on %1$s at %2$s', 'cmb-sqlite-form'),
            date_i18n(get_option('date_format')),
            date_i18n(get_option('time_format'))
        ) . '</small></p>';
        $message .= '</body></html>';

        // Send test email using SMTP_Mailer
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-smtp-mailer.php';
        $result = CMBSQF_SMTP_Mailer::send_test_email($config, $recipient, $subject, $message);

        // Prepare response
        $response_data = [
            'success'    => $result['success'],
            'debug_log'  => $result['debug_log'],
            'sent_data'  => [
                'to'      => $recipient,
                'subject' => $subject,
            ],
        ];

        if ($result['success']) {
            $response_data['message'] = sprintf(
                /* translators: %s: recipient email address */
                __('Test email sent successfully to %s. Please check your inbox.', 'cmb-sqlite-form'),
                $recipient
            );
            
            return rest_ensure_response($response_data);
        } else {
            // Email failed - include error details
            $response_data['message'] = __('Failed to send test email.', 'cmb-sqlite-form');
            
            if (isset($result['error_details'])) {
                $response_data['error_details'] = $result['error_details'];
            }

            // Return 500 status for SMTP failures
            return new WP_REST_Response($response_data, 500);
        }
    }
}
