<?php
/**
 * Blocklist REST Controller
 *
 * Purpose: Handle blocklist updates and source validation
 * Location: /rest/admin/class-blocklist-controller.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_REST_Blocklist {

    /**
     * Register REST API routes.
     */
    public static function register() {
        // POST /wp-json/cmbsqf/v1/admin/update-blocklist
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/update-blocklist',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'update_blocklist'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'form_id' => [
                        'required'          => false,
                        'type'              => 'integer',
                        'sanitize_callback' => 'absint',
                        'default'           => 0,
                    ],
                    'sources' => [
                        'required'          => true,
                        'type'              => 'array',
                        'validate_callback' => function($param) {
                            return is_array($param);
                        },
                    ],
                ],
            ]
        );

        // POST /wp-json/cmbsqf/v1/admin/validate-source
        register_rest_route(
            CMBSQF_Constants::REST_NAMESPACE,
            '/admin/validate-source',
            [
                'methods'             => 'POST',
                'callback'            => [__CLASS__, 'validate_source'],
                'permission_callback' => ['CMBSQF_Admin_Loader', 'check_admin_permission'],
                'args'                => [
                    'url' => [
                        'required'          => true,
                        'type'              => 'string',
                        'validate_callback' => function($param) {
                            return filter_var($param, FILTER_VALIDATE_URL) !== false;
                        },
                        'sanitize_callback' => 'esc_url_raw',
                    ],
                ],
            ]
        );
    }

    /**
     * POST /wp-json/cmbsqf/v1/admin/update-blocklist
     *
     * Update blocklist from sources.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public static function update_blocklist($request) {
        $form_id = $request->get_param('form_id');
        $sources = $request->get_param('sources');

        // Filter empty URLs
        $sources = array_filter($sources, function($url) {
            return !empty(trim($url));
        });

        // Allow empty sources - will create an empty blocklist file
        // Load blocklist manager
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-blocklist-manager.php';

        // Update blocklist (pass empty array if no sources)
        $result = CMBSQF_Blocklist_Manager::update_blocklist($form_id, $sources);

        // Always return 200 OK to avoid console errors
        return new WP_REST_Response($result, 200);
    }

    /**
     * Validate a source URL.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response Response object.
     */
    public static function validate_source($request) {
        $url = $request->get_param('url');
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-blocklist-manager.php';
        
        $result = CMBSQF_Blocklist_Manager::validate_source($url);
        
        if (is_wp_error($result)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $result->get_error_message(),
            ], 400);
        }
        
        return new WP_REST_Response([
            'success' => true,
            'domains' => $result,
            /* translators: %d: number of domains found in the source */
            'message' => sprintf(__('%d domains found', 'cmb-sqlite-form'), $result),
        ], 200);
    }
}
