<?php
/**
 * Database Submissions Manager
 * 
 * Purpose: Manage form submission entries in the database
 * Location: /includes/db/class-db-submissions.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_DB_Submissions {
    
    /**
     * Create a new submission entry (save form submission).
     *
     * @param array $data Submission data containing:
     *                    - form_id (int): Form ID (required)
     *                    - name, surname, email, phone, organization, position, website, subject, message (string)
     *                    - privacy_status (string): 'accepted' or 'not_configured'
     *                    - submitted_at (string): MySQL datetime format (optional, defaults to current time)
     * @return int|false Submission ID on success, false on failure.
     */
    public static function create($data) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $form_id = intval($data['form_id']);
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return false;
        }
        
        // Prepare data
        $insert_data = [
            'name'           => $data['name'] ?? '',
            'surname'        => $data['surname'] ?? '',
            'email'          => $data['email'] ?? '',
            'phone'          => $data['phone'] ?? '',
            'organization'   => $data['organization'] ?? '',
            'position'       => $data['position'] ?? '',
            'website'        => $data['website'] ?? '',
            'subject'        => $data['subject'] ?? '',
            'message'        => $data['message'] ?? '',
            'privacy_status' => $data['privacy_status'] ?? 'not_configured',
            'ip_address'     => self::get_client_ip(),
            'user_agent'     => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
            'created_at'     => $data['submitted_at'] ?? current_time('mysql'),
        ];
        
        // Build INSERT query
        $fields = array_keys($insert_data);
        $placeholders = array_fill(0, count($fields), '?');
        
        $sql = sprintf(
            "INSERT INTO entries (%s) VALUES (%s)",
            implode(', ', $fields),
            implode(', ', $placeholders)
        );
        
        $stmt = $db->prepare($sql);
        
        if (!$stmt) {
            return false;
        }
        
        // Bind values
        $i = 1;
        foreach ($insert_data as $value) {
            $stmt->bindValue($i++, $value, SQLITE3_TEXT);
        }
        
        $success = $stmt->execute();
        
        if (!$success) {
            return false;
        }
        
        return $db->lastInsertRowID();
    }
    
    /**
     * Get client IP address
     * 
     * @return string IP address
     */
    private static function get_client_ip() {
        $ip_keys = [
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ];
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER)) {
                $ip = sanitize_text_field(wp_unslash($_SERVER[$key]));
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return '0.0.0.0';
    }
    
    /**
     * Get all submissions for a form (no pagination)
     * 
     * @param int $form_id Form ID
     * @return array Array of submissions
     */
    public static function get_all($form_id) {
        // Use a large limit to get all entries
        return self::get_by_form($form_id, 100000, 0, 'created_at', 'DESC', '');
    }

    /**
     * Get submissions for a form
     * 
     * @param int    $form_id Form ID
     * @param int    $limit   Limit
     * @param int    $offset  Offset
     * @param string $orderby Order by column
     * @param string $order   Order direction (ASC/DESC)
     * @param string $search  Search query
     * @return array Array of submissions
     */
    public static function get_by_form($form_id, $limit = 50, $offset = 0, $orderby = 'created_at', $order = 'DESC', $search = '') {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return [];
        }
        
        
        // Build WHERE clause for search
        $where = '';
        $search_params = [];
        if (!empty($search)) {
            $where = " WHERE (name LIKE ? OR surname LIKE ? OR email LIKE ? OR message LIKE ?)";
            $search_term = '%' . $search . '%';
            $search_params = [$search_term, $search_term, $search_term, $search_term];
        }
        
        // Validate order direction
        $order = strtoupper($order) === 'ASC' ? 'ASC' : 'DESC';
        
        // Validate orderby
        $allowed_orderby = ['id', 'name', 'email', 'created_at'];
        if (!in_array($orderby, $allowed_orderby, true)) {
            $orderby = 'created_at';
        }
        
        $sql = "SELECT * FROM entries" . $where . " ORDER BY " . $orderby . " " . $order . " LIMIT ? OFFSET ?";
        $stmt = $db->prepare($sql);
        
        if (!$stmt) {
            return [];
        }
        
        // Bind search parameters if applicable
        $param_index = 1;
        foreach ($search_params as $param) {
            $stmt->bindValue($param_index++, $param, SQLITE3_TEXT);
        }
        
        // Bind LIMIT and OFFSET
        $stmt->bindValue($param_index++, $limit, SQLITE3_INTEGER);
        $stmt->bindValue($param_index++, $offset, SQLITE3_INTEGER);
        
        $result = $stmt->execute();
        $submissions = [];
        
        while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $submissions[] = $row;
        }
        
        return $submissions;
    }
    
    /**
     * Get a single entry by ID
     * 
     * @param int $entry_id Entry ID
     * @param int $form_id Form ID (required to prevent cross-form access)
     * @return array|null Entry data or null if not found
     */
    public static function get_by_id($entry_id, $form_id) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return null;
        }
        
        $stmt = $db->prepare("SELECT * FROM entries WHERE id = ?");
        
        if (!$stmt) {
            return null;
        }
        
        $stmt->bindValue(1, $entry_id, SQLITE3_INTEGER);
        $result = $stmt->execute();
        
        if ($row = $result->fetchArray(SQLITE3_ASSOC)) {
            $row['form_id'] = $form_id;
            return $row;
        }
        
        return null;
    }
    
    /**
     * Update entry fields
     * 
     * @param int   $entry_id Entry ID
     * @param int   $form_id  Form ID (required for security)
     * @param array $data     Data to update
     * @return bool Success
     */
    public static function update($entry_id, $form_id, $data) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return false;
        }
        
        // Build UPDATE query with allowed fields
        $allowed_fields = [
            'name',
            'surname', 
            'email',
            'phone',
            'organization',
            'position',
            'website',
            'subject',
            'message',
            'privacy_status',
            'email_delivery',
            'email_delivery_error'
        ];
        
        $update_fields = [];
        $values = [];
        
        foreach ($data as $key => $value) {
            if (in_array($key, $allowed_fields, true)) {
                $update_fields[] = "$key = ?";
                $values[] = $value;
            }
        }
        
        if (empty($update_fields)) {
            return false;
        }
        
        $sql = "UPDATE entries SET " . implode(', ', $update_fields) . " WHERE id = ?";
        $stmt = $db->prepare($sql);
        
        if (!$stmt) {
            return false;
        }
        
        // Bind values
        $i = 1;
        foreach ($values as $value) {
            $stmt->bindValue($i++, $value, SQLITE3_TEXT);
        }
        $stmt->bindValue($i, $entry_id, SQLITE3_INTEGER);
        
        return $stmt->execute() !== false;
    }
    
    /**
     * Get count of submissions for a form
     * 
     * @param int    $form_id Form ID
     * @param string $search  Search query
     * @return int Count of submissions
     */
    public static function get_count($form_id, $search = '') {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return 0;
        }
        
        
        // Build WHERE clause for search
        $where = '';
        $search_params = [];
        if (!empty($search)) {
            $where = " WHERE (name LIKE ? OR surname LIKE ? OR email LIKE ? OR message LIKE ?)";
            $search_term = '%' . $search . '%';
            $search_params = [$search_term, $search_term, $search_term, $search_term];
        }
        
        $sql = "SELECT COUNT(*) as total FROM entries" . $where;
        
        if (!empty($search_params)) {
            $stmt = $db->prepare($sql);
            if (!$stmt) {
                return 0;
            }
            
            $param_index = 1;
            foreach ($search_params as $param) {
                $stmt->bindValue($param_index++, $param, SQLITE3_TEXT);
            }
            
            $result = $stmt->execute();
            $row = $result->fetchArray(SQLITE3_ASSOC);
            return (int) ($row['total'] ?? 0);
        }
        
        $result = $db->querySingle($sql);
        
        return (int) $result;
    }
    
    /**
     * Update email delivery status for a submission
     * 
     * @param int    $form_id       Form ID
     * @param int    $submission_id Submission ID
     * @param string $status        Status: 'sent', 'failed', 'not_configured'
     * @param string $error         Error message if failed
     * @return bool Success
     */
    public static function update_email_delivery($form_id, $submission_id, $status, $error = null) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return false;
        }
        
        $stmt = $db->prepare("UPDATE entries SET email_delivery = ?, email_delivery_error = ? WHERE id = ?");
        
        if (!$stmt) {
            return false;
        }
        
        $stmt->bindValue(1, $status, SQLITE3_TEXT);
        $stmt->bindValue(2, $error, SQLITE3_TEXT);
        $stmt->bindValue(3, $submission_id, SQLITE3_INTEGER);
        
        return $stmt->execute() !== false;
    }
    
    /**
     * Delete a submission
     * 
     * @param int $entry_id Entry ID
     * @param int $form_id  Form ID (required for security)
     * @return bool Success
     */
    public static function delete($entry_id, $form_id) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return false;
        }
        
        $stmt = $db->prepare("DELETE FROM entries WHERE id = ?");
        
        if (!$stmt) {
            return false;
        }
        
        $stmt->bindValue(1, $entry_id, SQLITE3_INTEGER);
        
        return $stmt->execute() !== false;
    }
    
    /**
     * Delete all submissions for a form
     * 
     * @param int $form_id Form ID
     * @return int|false Number of deleted entries or false on failure
     */
    public static function delete_all_by_form($form_id) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        
        $db = CMBSQF_DB_Core::get_form_entries_db($form_id);
        
        if (!$db || is_wp_error($db)) {
            return false;
        }
        
        // Get count before deleting
        $count = $db->querySingle("SELECT COUNT(*) FROM entries");
        
        if ($count === false) {
            return false;
        }
        
        // Delete all entries
        $result = $db->exec("DELETE FROM entries");
        
        if ($result === false) {
            return false;
        }
        
        return (int) $count;
    }
}
