<?php
/**
 * Settings Operations for CMB SQLite Form
 *
 * Purpose: CRUD operations for plugin settings
 * Location: /includes/db/class-db-settings.php
 *
 * Handles:
 * - Get settings with override logic (form overrides global)
 * - Get form-specific settings (direct access)
 * - Get global settings
 * - Set form-specific settings
 * - Set global settings
 *
 * Loaded by: cmb-sqlite-form.php (contexts that need settings access)
 * Dependencies: CMBSQF_DB_Core (loaded below)
 */

if (!defined('ABSPATH')) {
	exit;
}

// Settings depends on Core for SQLite storage
require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';

class CMBSQF_DB_Settings {

	/**
	 * Get setting value with automatic override handling.
	 *
	 * For form-specific requests (form_id > 0):
	 * - Checks the section's override flag (e.g., form__fields__override)
	 * - If override is '0', falls back to global settings
	 * - If override is '1', uses form-specific settings
	 *
	 * For global requests (form_id = 0):
	 * - Returns global value directly
	 *
	 * @param string $key     Full setting key (e.g., 'form__ui_texts__header_html').
	 * @param int    $form_id Form ID (0 for global).
	 * @param string $default Default value if not found.
	 * @return string Setting value.
	 */
	public static function get($key, $form_id = 0, $default = '') {
		// Global settings - return directly
		if ($form_id === 0) {
			return self::get_global_setting($key, $default);
		}

		// Form-specific settings - check override flag
		// Extract section key from full key
		// Keys have format: prefix__section__field or prefix__section__subfield__property
		// Section override is always at: prefix__section__override
		// Examples:
		//   - notifications__to_user__enabled → section: notifications__to_user
		//   - form__fields__name_enabled → section: form__fields
		$parts = explode('__', $key);
		if (count($parts) >= 3) {
			// Take first two parts as section key
			$section_key = $parts[0] . '__' . $parts[1];
			$override_key = $section_key . '__override';
			
			// Check if override is enabled for this section
			$override = self::get_form_setting($override_key, $form_id, '0');
			
			// If override is disabled, read from global
			if ($override === '0') {
				return self::get_global_setting($key, $default);
			}
		}

		// Override is enabled (or key doesn't match pattern) - read from form
		return self::get_form_setting($key, $form_id, $default);
	}

	/**
	 * Set setting value (simple wrapper).
	 *
	 * Sets form-specific value if form_id provided, otherwise global value.
	 *
	 * @param string $key     Setting key.
	 * @param string $value   Setting value.
	 * @param int    $form_id Form ID (0 for global).
	 * @return bool Success.
	 */
	public static function set($key, $value, $form_id = 0) {
		if ($form_id > 0) {
			return self::set_form_setting($key, $value, $form_id);
		}
		return self::set_global_setting($key, $value);
	}

	/**
	 * Get form-specific setting (direct access, no override logic).
	 *
	 * @param string $key     Setting key.
	 * @param int    $form_id Form ID.
	 * @param string $default Default value.
	 * @return string Setting value.
	 */
	public static function get_form_setting($key, $form_id, $default = '') {
		$db = CMBSQF_DB_Core::get_form_settings_db($form_id);
		return self::get_setting_from_db($db, $key, $default);
	}

	/**
	 * Get global setting.
	 *
	 * @param string $key     Setting key.
	 * @param string $default Default value.
	 * @return string Setting value.
	 */
	public static function get_global_setting($key, $default = '') {
		$db = CMBSQF_DB_Core::get_global_db();
		return self::get_setting_from_db($db, $key, $default);
	}

	/**
	 * Get setting value from a database connection.
	 *
	 * @param SQLite3|WP_Error $db      Database connection.
	 * @param string           $key     Setting key.
	 * @param string           $default Default value.
	 * @return string Setting value.
	 */
	private static function get_setting_from_db($db, $key, $default = '') {
		if (!$db || is_wp_error($db)) {
			return $default;
		}

		$stmt = $db->prepare('SELECT setting_value FROM settings WHERE setting_key = :key LIMIT 1');
		if (!$stmt) {
			$db->close();
			return $default;
		}

		$stmt->bindValue(':key', $key, SQLITE3_TEXT);
		$result = $stmt->execute();
		$row    = $result ? $result->fetchArray(SQLITE3_ASSOC) : null;

		$stmt->close();
		$db->close();

		return $row ? $row['setting_value'] : $default;
	}

	/**
	 * Set form-specific setting.
	 *
	 * @param string $key     Setting key.
	 * @param string $value   Setting value.
	 * @param int    $form_id Form ID.
	 * @return bool Success.
	 */
	public static function set_form_setting($key, $value, $form_id) {
		CMBSQF_DB_Core::create_form_settings_db($form_id); // Ensure DB exists
		$db = CMBSQF_DB_Core::get_form_settings_db($form_id);
		return self::set_setting_in_db($db, $key, $value);
	}

	/**
	 * Set global setting.
	 *
	 * @param string $key   Setting key.
	 * @param string $value Setting value.
	 * @return bool Success.
	 */
	public static function set_global_setting($key, $value) {
		CMBSQF_DB_Core::create_global_db(); // Ensure DB exists
		$db = CMBSQF_DB_Core::get_global_db();
		return self::set_setting_in_db($db, $key, $value);
	}

	/**
	 * Set setting value in a database connection.
	 *
	 * @param SQLite3|WP_Error $db    Database connection.
	 * @param string           $key   Setting key.
	 * @param string           $value Setting value.
	 * @return bool Success.
	 */
	private static function set_setting_in_db($db, $key, $value) {
		if (!$db || is_wp_error($db)) {
			return false;
		}

		$stmt = $db->prepare('INSERT OR REPLACE INTO settings (setting_key, setting_value) VALUES (:key, :value)');
		if (!$stmt) {
			$db->close();
			return false;
		}

		$stmt->bindValue(':key', $key, SQLITE3_TEXT);
		$stmt->bindValue(':value', (string) $value, SQLITE3_TEXT);
		$result = $stmt->execute();

		$stmt->close();
		$db->close();

		return (bool) $result;
	}
}
