<?php
/**
 * Core SQLite Storage for CMB SQLite Form
 *
 * Purpose: Low-level SQLite storage management
 * Location: /includes/db/class-db-core.php
 *
 * Handles:
 * - Database connections (form and global)
 * - Schema creation and initialization
 * - Path resolution and validation
 * - Directory protection
 *
 * Loaded by: class-db-settings.php and class-db-entries.php
 * Dependencies: class-installer.php (for schema creation)
 */

if (!defined('ABSPATH')) {
	exit;
}

require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core-installer.php';

class CMBSQF_DB_Core {

	// Configuration
	private const DB_DIR             = 'cmb-sqlite-form';
	private const FORMS_SUBDIR       = 'forms';
	private const SETTINGS_DB_FILE   = 'settings.db';
	private const ENTRIES_DB_FILE    = 'entries.db';
	private const GLOBAL_DB_FILE     = 'global.db';

	// ================================================================
	// DATABASE CONNECTIONS
	// ================================================================

	/**
	 * Get form settings database connection.
	 *
	 * @param int $form_id Form ID.
	 * @return SQLite3|WP_Error Database connection or error.
	 */
	public static function get_form_settings_db($form_id) {
		$validated = self::validate_form_id($form_id);
		if (is_wp_error($validated)) {
			return $validated;
		}

		return self::get_db(self::get_form_settings_db_path($validated));
	}

	/**
	 * Create form settings database.
	 *
	 * @param int $form_id Form ID.
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public static function create_form_settings_db($form_id) {
		$validated = self::validate_form_id($form_id);
		if (is_wp_error($validated)) {
			return $validated;
		}

		return self::create_db(
			self::get_form_settings_db_path($validated),
			[CMBSQF_DB_Core_Installer::class, 'create_form_settings_schema'],
			$validated
		);
	}

	/**
	 * Get form entries database connection.
	 *
	 * @param int $form_id Form ID.
	 * @return SQLite3|WP_Error Database connection or error.
	 */
	public static function get_form_entries_db($form_id) {
		$validated = self::validate_form_id($form_id);
		if (is_wp_error($validated)) {
			return $validated;
		}

		return self::get_db(self::get_form_entries_db_path($validated));
	}

	/**
	 * Create form entries database.
	 *
	 * @param int $form_id Form ID.
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public static function create_form_entries_db($form_id) {
		$validated = self::validate_form_id($form_id);
		if (is_wp_error($validated)) {
			return $validated;
		}

		return self::create_db(
			self::get_form_entries_db_path($validated),
			[CMBSQF_DB_Core_Installer::class, 'create_form_entries_schema']
		);
	}

	/**
	 * Get global database connection.
	 *
	 * Creates the database automatically if it doesn't exist.
	 *
	 * @return SQLite3|WP_Error Database connection or error.
	 */
	public static function get_global_db() {
		$path = self::get_global_db_path();
		
		// If database doesn't exist, create it first
		if (!is_wp_error($path) && !file_exists($path)) {
			self::create_global_db();
		}
		
		return self::get_db($path);
	}

	/**
	 * Create global database.
	 *
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public static function create_global_db() {
		$result = self::create_db(
			self::get_global_db_path(),
			[CMBSQF_DB_Core_Installer::class, 'create_global_schema']
		);

		// Initialize with default CSS template
		if ($result === true) {
			$default_css_path = CMBSQF_PLUGIN_DIR . 'admin/css-templates/default.css';
			if (file_exists($default_css_path)) {
				$default_css = file_get_contents($default_css_path);
				if ($default_css !== false) {
					require_once CMBSQF_PLUGIN_DIR . 'includes/class-css-storage.php';
					CMBSQF_CSS_Storage::save($default_css, 0);
				}
			}
			
			// Initialize global defaults immediately after creation
			self::initialize_global_defaults();
		}

		return $result;
	}

	/**
	 * Open SQLite connection.
	 *
	 * @param string $path Database file path.
	 * @return SQLite3|WP_Error Connection or error.
	 */
	private static function open_sqlite($path) {
		if (!class_exists('SQLite3')) {
			return new WP_Error('sqlite_missing', __('SQLite3 extension not available.', 'cmb-sqlite-form'));
		}

		try {
			$db = new SQLite3($path);
			$db->busyTimeout(5000);
			return $db;
		} catch (Exception $e) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log('CMBSQF SQLite Error: ' . $e->getMessage());
			return new WP_Error('sqlite_error', __('Database error occurred.', 'cmb-sqlite-form'));
		}
	}

	/**
	 * Generic database connection handler.
	 *
	 * @param string|WP_Error $path Database path.
	 * @return SQLite3|WP_Error Database connection or error.
	 */
	private static function get_db($path) {
		if (is_wp_error($path)) {
			return $path;
		}

		if (!file_exists($path)) {
			return new WP_Error('db_not_found', __('Database does not exist.', 'cmb-sqlite-form'));
		}

		return self::open_sqlite($path);
	}

	/**
	 * Validate and sanitize form ID.
	 *
	 * @param int $form_id Form ID to validate.
	 * @return int|WP_Error Validated form ID or error.
	 */
	private static function validate_form_id($form_id) {
		$form_id = intval($form_id);
		if ($form_id <= 0) {
			return new WP_Error('invalid_form_id', __('Invalid form ID.', 'cmb-sqlite-form'));
		}
		return $form_id;
	}

	/**
	 * Generic database creation handler.
	 *
	 * @param string|WP_Error $db_path         Database path.
	 * @param callable        $schema_callback Callback to create schema.
	 * @param mixed           ...$args         Additional arguments for schema callback.
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	private static function create_db($db_path, $schema_callback, ...$args) {
		if (is_wp_error($db_path)) {
			return $db_path;
		}

		// Ensure directory exists and is protected
		$dir = dirname($db_path);
		wp_mkdir_p($dir);
		CMBSQF_DB_Core_Installer::protect_directory($dir);

		$db = self::open_sqlite($db_path);
		if (is_wp_error($db)) {
			return $db;
		}

		// Execute schema callback with additional arguments
		call_user_func($schema_callback, $db, ...$args);

		$db->close();
		return true;
	}

	// ================================================================
	// PATH MANAGEMENT
	// ================================================================

	/**
	 * Generic path builder for form databases.
	 *
	 * @param int    $form_id Form ID.
	 * @param string $db_file Database filename.
	 * @return string|WP_Error Database path or error.
	 */
	private static function get_form_db_path($form_id, $db_file) {
		$base_dir = self::get_data_directory();
		if (is_wp_error($base_dir)) {
			return $base_dir;
		}

		return trailingslashit($base_dir) . self::FORMS_SUBDIR . '/' . $form_id . '/' . $db_file;
	}

	/**
	 * Get path to form settings database file.
	 *
	 * @param int $form_id Form ID.
	 * @return string|WP_Error Database path or error.
	 */
	public static function get_form_settings_db_path($form_id) {
		return self::get_form_db_path($form_id, self::SETTINGS_DB_FILE);
	}

	/**
	 * Get path to form entries database file.
	 *
	 * @param int $form_id Form ID.
	 * @return string|WP_Error Database path or error.
	 */
	public static function get_form_entries_db_path($form_id) {
		return self::get_form_db_path($form_id, self::ENTRIES_DB_FILE);
	}

	/**
	 * Get path to global database file.
	 *
	 * @return string|WP_Error Database path or error.
	 */
	public static function get_global_db_path() {
		$base_dir = self::get_data_directory();
		if (is_wp_error($base_dir)) {
			return $base_dir;
		}

		$global_dir = trailingslashit($base_dir) . 'global';
		
		// Ensure global directory exists
		if (!is_dir($global_dir)) {
			wp_mkdir_p($global_dir);
		}
		
		return trailingslashit($global_dir) . self::GLOBAL_DB_FILE;
	}

	    /**
     * Get the base data directory path.
     * 
     * Using WordPress uploads directory for database storage.
     * All database files are stored in uploads/cmb-sqlite-form/:
     * - forms/{form_id}/ - per-form databases
     * - global/ - global settings database
     *
     * @return string Absolute path to data directory.
     */
    public static function get_data_directory() {
        $upload_dir = wp_upload_dir();
        $data_dir = $upload_dir['basedir'] . '/cmb-sqlite-form';
        
        // Ensure base directory exists
        if (!is_dir($data_dir)) {
            wp_mkdir_p($data_dir);
            // Protect base directory
            require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core-installer.php';
            CMBSQF_DB_Core_Installer::protect_directory($data_dir);
        }
        
        return $data_dir;
    }


	// ================================================================
	// FORM MANAGEMENT
	// ================================================================

	/**
	 * Get all forms.
	 *
	 * Scans the forms directory and reads settings.db for each form.
	 *
	 * @return array List of forms with id, name, shortcode, num_entries.
	 */
	public static function get_all_forms() {
		$base_dir = self::get_data_directory();
		if (is_wp_error($base_dir)) {
			return [];
		}

		$forms_dir = trailingslashit($base_dir) . self::FORMS_SUBDIR;
		if (!is_dir($forms_dir)) {
			return [];
		}

		$forms = [];
		$dirs = glob($forms_dir . '/*', GLOB_ONLYDIR);

		if ($dirs) {
			foreach ($dirs as $dir) {
				$form_id = basename($dir);
				if (!is_numeric($form_id)) {
					continue;
				}

				$form_id = intval($form_id);

				// Get form name from settings
				$name = 'Form #' . $form_id;
				$settings_db = self::get_form_settings_db($form_id);
				if (!is_wp_error($settings_db)) {
					$result = $settings_db->querySingle("SELECT setting_value FROM settings WHERE setting_key = 'form_name'");
					if ($result) {
						$name = $result;
					}
					$settings_db->close();
				}

				// Get entry count
				$num_entries = 0;
				$entries_db = self::get_form_entries_db($form_id);
				if (!is_wp_error($entries_db)) {
					$count = $entries_db->querySingle("SELECT COUNT(*) FROM entries");
					if ($count !== false) {
						$num_entries = $count;
					}
					$entries_db->close();
				}

				$forms[] = [
					'id'          => $form_id,
					'name'        => $name,
					'shortcode'   => '[cmb_sqlite_form id="' . $form_id . '"]',
					'num_entries' => $num_entries,
				];
			}
		}

		return $forms;
	}

	/**
	 * Create a new form.
	 *
	 * @param string $name Form name.
	 * @return int|WP_Error New form ID or error.
	 */
	public static function create_form($name) {
		$base_dir = self::get_data_directory();
		if (is_wp_error($base_dir)) {
			return $base_dir;
		}

		$forms_dir = trailingslashit($base_dir) . self::FORMS_SUBDIR;
		if (!is_dir($forms_dir)) {
			wp_mkdir_p($forms_dir);
		}

		// Find next available ID
		$existing_ids = [];
		$dirs = glob($forms_dir . '/*', GLOB_ONLYDIR);
		if ($dirs) {
			foreach ($dirs as $dir) {
				$id = basename($dir);
				if (is_numeric($id)) {
					$existing_ids[] = intval($id);
				}
			}
		}

		$new_id = 1;
		if (!empty($existing_ids)) {
			$new_id = max($existing_ids) + 1;
		}

		// Create form directory
		$form_dir = trailingslashit($forms_dir) . $new_id;
		if (!wp_mkdir_p($form_dir)) {
			return new WP_Error('dir_create_failed', __('Failed to create form directory.', 'cmb-sqlite-form'));
		}

		// Protect directory
		CMBSQF_DB_Core_Installer::protect_directory($form_dir);

		// Initialize databases
		$settings_result = self::create_form_settings_db($new_id);
		if (is_wp_error($settings_result)) {
			return $settings_result;
		}

		$entries_result = self::create_form_entries_db($new_id);
		if (is_wp_error($entries_result)) {
			return $entries_result;
		}

		// Save form name and disable overrides (use global settings)
	$db = self::get_form_settings_db($new_id);
	if (!is_wp_error($db)) {
		$default_settings = [
			'form_name' => $name,
			
			// Disable all overrides - new forms inherit from global settings
			'form__ui_texts__override' => '0',
			'form__fields__override' => '0',
			'form__privacy__override' => '0',
			'form__styling__override' => '0',
			'form__bot_protection__override' => '0',
			'form__junk_email__override' => '0',
			'notifications__outgoing_server__override' => '0',			
			'notifications__to_user__override' => '0',
			'notifications__to_admin__override' => '0',
		];
		
		$stmt = $db->prepare('INSERT OR REPLACE INTO settings (setting_key, setting_value) VALUES (:key, :value)');
		foreach ($default_settings as $key => $value) {
			$stmt->bindValue(':key', $key, SQLITE3_TEXT);
			$stmt->bindValue(':value', $value, SQLITE3_TEXT);
			$stmt->execute();
			$stmt->reset();
		}
		$db->close();
	}

	// Initialize with default CSS template
	$default_css_path = CMBSQF_PLUGIN_DIR . 'admin/css-templates/default.css';
	if (file_exists($default_css_path)) {
		$default_css = file_get_contents($default_css_path);
		if ($default_css !== false) {
			require_once CMBSQF_PLUGIN_DIR . 'includes/class-css-storage.php';
			CMBSQF_CSS_Storage::save($default_css, $new_id);
		}
	}

	return $new_id;
	}

	/**
	 * Rename a form.
	 *
	 * @param int    $form_id Form ID.
	 * @param string $name    New name.
	 * @return bool True on success.
	 */
	public static function rename_form($form_id, $name) {
		$db = self::get_form_settings_db($form_id);
		if (is_wp_error($db)) {
			return false;
		}

		$stmt = $db->prepare('INSERT OR REPLACE INTO settings (setting_key, setting_value) VALUES (:key, :value)');
		$stmt->bindValue(':key', 'form_name', SQLITE3_TEXT);
		$stmt->bindValue(':value', $name, SQLITE3_TEXT);
		$result = $stmt->execute();
		$db->close();

		return $result !== false;
	}

	/**
	 * Delete a form.
	 *
	 * @param int $form_id Form ID.
	 * @return bool True on success.
	 */
	public static function delete_form($form_id) {
		$validated = self::validate_form_id($form_id);
		if (is_wp_error($validated)) {
			return false;
		}

		$base_dir = self::get_data_directory();
		if (is_wp_error($base_dir)) {
			return false;
		}

		$form_dir = trailingslashit($base_dir) . self::FORMS_SUBDIR . '/' . $validated;

		if (!is_dir($form_dir)) {
			return false; // Already gone
		}

		// Recursive delete
		return self::recursive_rmdir($form_dir);
	}

	/**
	 * Recursively delete a directory.
	 *
	 * @param string $dir Directory path.
	 * @return bool True on success.
	 */
	private static function recursive_rmdir($dir) {
		if (!is_dir($dir)) {
			return false;
		}

		$files = array_diff(scandir($dir), ['.', '..']);
		foreach ($files as $file) {
			$path = $dir . '/' . $file;
			// phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Direct unlink needed for recursive directory cleanup
			(is_dir($path)) ? self::recursive_rmdir($path) : unlink($path);
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir -- Direct rmdir needed for recursive cleanup
		return rmdir($dir);
	}

	/**
	 * Initialize global default settings.
	 * 
	 * This is the single source of truth for all default global settings.
	 * Called by:
	 * - create_form() when creating the first form
	 * - CMBSQF_DB_Core_Installer::seed_global_defaults() during DB creation
	 */
	public static function initialize_global_defaults() {
		$db = self::get_global_db();
		if (is_wp_error($db)) {
			return;
		}

		$defaults = [
			// UI Texts (4 fields)
			'form__ui_texts__header_html' => '<h2>Contact Us</h2>',
			'form__ui_texts__success_text' => 'Thank you for your message. We will contact you soon.',
			'form__ui_texts__error_text' => 'An error occurred. Please try again.',
			'form__ui_texts__submit_text' => 'Send Message',
			
			// Fields - All 9 fields with enabled + required (18 total)
		'form__fields__name_enabled' => '1',
		'form__fields__name_required' => '1',
		'form__fields__surname_enabled' => '0',
		'form__fields__surname_required' => '0',
		'form__fields__email_enabled' => '1',
		'form__fields__email_required' => '1',
		'form__fields__phone_enabled' => '0',
		'form__fields__phone_required' => '0',
		'form__fields__organization_enabled' => '0',
		'form__fields__organization_required' => '0',
		'form__fields__website_enabled' => '0',
		'form__fields__website_required' => '0',
		'form__fields__position_enabled' => '0',
		'form__fields__position_required' => '0',
		'form__fields__subject_enabled' => '0',
		'form__fields__subject_required' => '0',
		'form__fields__message_enabled' => '1',
		'form__fields__message_required' => '0',
			
			// Bot Protection (7 fields)
			'form__bot_protection__honeypot' => '1',
			'form__bot_protection__honeypot_field' => 'business_url',
			'form__bot_protection__min_time_enabled' => '1',
			'form__bot_protection__min_time' => '3',
			'form__bot_protection__captcha_enabled' => '0',
			'form__bot_protection__captcha_question' => 'What is 2 + 2?',
		'form__bot_protection__captcha_answer' => '4',
			
			// Behavior (1 field)
			'form__behavior__default_form_name_template' => 'Form {id}',
			
			// Styling (2 fields)
			'form__styling__template' => 'default',
			'form__styling__custom_css' => '',
			
			// Privacy (2 fields)
		'form__privacy__enabled' => '1',
		'form__privacy__html' => 'I accept the <a href="{url}">Privacy Policy</a>',
			
			// Junk Email (6 fields)
		'form__junk_email__block_disposable' => '0',
		'form__junk_email__disposable_message' => 'Disposable or temporary email addresses are not allowed. Please use a permanent email address.',
		'form__junk_email__sources' => '[]',
		'form__junk_email__sources_metadata' => '[]',
		'form__junk_email__total_domains' => '0',
		'form__junk_email__last_update' => '0',
			
			// Notifications - Sender (2 fields)
			'notifications__sender__from_name' => get_option('blogname'),
			'notifications__sender__from_email' => get_option('admin_email'),
			
			// Notifications - To User (3 fields)
			'notifications__to_user__enabled' => '1',
			'notifications__to_user__subject' => 'Thank you for contacting {site_name}',
			'notifications__to_user__message' => "Dear {name},\n\nThank you for contacting us. We have received your message and will get back to you as soon as possible.\n\nBest regards,\n{site_name}",
			
			// Notifications - To Admin (4 fields)
			'notifications__to_admin__enabled' => '1',
			'notifications__to_admin__emails' => get_option('admin_email'),
			'notifications__to_admin__subject' => 'New form submission from {name}',
			'notifications__to_admin__message' => "New submission on {form_name}:\n\n<strong>Name:</strong> {name} {surname}\n<strong>Email:</strong> {email}\n<strong>Phone:</strong> {phone}\n\n<strong>Message:</strong>\n{message}\n\n---\nSent from {site_name}",
			
			// Notifications - Outgoing Server (6 fields)
			'notifications__outgoing_server__smtp_enabled' => '0',
			'notifications__outgoing_server__smtp_host' => '',
			'notifications__outgoing_server__smtp_port' => '587',
			'notifications__outgoing_server__smtp_username' => '',
			'notifications__outgoing_server__smtp_password' => '',
			'notifications__outgoing_server__smtp_encryption' => 'tls',
			
			// General - Data (1 field)
			'general__data__delete_on_uninstall' => '0',
		];

		$stmt = $db->prepare('INSERT OR IGNORE INTO settings (setting_key, setting_value) VALUES (:key, :value)');
		foreach ($defaults as $key => $value) {
			$stmt->bindValue(':key', $key, SQLITE3_TEXT);
			$stmt->bindValue(':value', $value, SQLITE3_TEXT);
			$stmt->execute();
			$stmt->reset();
		}
		$db->close();
	}

}
