<?php
/**
 * Database Core Installer
 *
 * Purpose: Database schema creation and default seeding
 * Location: /includes/db/class-db-core-installer.php
 *
 * This class handles:
 * - Creating database tables
 * - Seeding default settings
 * - Directory protection (.htaccess, index.php)
 *
 * Loaded by: class-db-core.php (when creating databases)
 * Dependencies: None
 */

if (!defined('ABSPATH')) {
	exit;
}

class CMBSQF_DB_Core_Installer {

	/**
	 * Create form settings database schema.
	 *
	 * @param SQLite3 $db      Database connection.
	 * @param int     $form_id Form ID.
	 */
	public static function create_form_settings_schema($db, $form_id) {
		$db->exec(
			"CREATE TABLE IF NOT EXISTS settings (
                setting_key TEXT PRIMARY KEY,
                setting_value TEXT NOT NULL
            )"
		);
		self::seed_form_defaults($db, $form_id);
	}

	/**
	 * Create form entries database schema.
	 *
	 * @param SQLite3 $db Database connection.
	 */
	public static function create_form_entries_schema($db) {
		$db->exec(
			"CREATE TABLE IF NOT EXISTS entries (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT,
                surname TEXT,
                email TEXT,
                phone TEXT,
                organization TEXT,
                position TEXT,
                website TEXT,
                subject TEXT,
                message TEXT,
                privacy_status TEXT DEFAULT 'not_configured',
                ip_address TEXT NOT NULL,
                user_agent TEXT,
                created_at TEXT NOT NULL,
                email_delivery TEXT DEFAULT 'not_configured',
                email_delivery_error TEXT
            )"
		);
		$db->exec('CREATE INDEX IF NOT EXISTS idx_entries_created_at ON entries(created_at)');
		$db->exec('CREATE INDEX IF NOT EXISTS idx_entries_email ON entries(email)');
	}

	/**
	 * Create global database schema.
	 *
	 * @param SQLite3 $db Database connection.
	 */
	public static function create_global_schema($db) {
		$db->exec(
			"CREATE TABLE IF NOT EXISTS settings (
                setting_key TEXT PRIMARY KEY,
                setting_value TEXT NOT NULL
            )"
		);
		
		// Close installer's connection before initializing defaults
		$db->close();
		
		// Initialize defaults using the centralized method
		CMBSQF_DB_Core::initialize_global_defaults();
	}

	/**
	 * Seed default form settings.
	 *
	 * @param SQLite3 $db      Database connection.
	 * @param int     $form_id Form ID.
	 */
	private static function seed_form_defaults($db, $form_id) {
		$defaults = array(
			'form_name'                                    => 'Form #' . $form_id,
			// Override flags
			'form__fields__override'                       => '0',
			'form__ui_texts__override'                     => '0',
			'form__privacy__override'                      => '0',
			'form__styling__override'                      => '0',
			'form__bot_protection__override'               => '0',
			'form__junk_email__override'                   => '0',
			'notifications__to_user__override'             => '0',
			'notifications__to_admin__override'            => '0',
			'notifications__outgoing_server__override'     => '0',
		);

		self::insert_defaults($db, $defaults);
	}

	/**
	 * Insert default settings into database.
	 *
	 * @param SQLite3 $db       Database connection.
	 * @param array   $defaults Array of key => value pairs.
	 */
	private static function insert_defaults($db, array $defaults) {
		$stmt = $db->prepare('INSERT OR IGNORE INTO settings (setting_key, setting_value) VALUES (:key, :value)');
		if (!$stmt) {
			return;
		}

		foreach ($defaults as $key => $value) {
			$stmt->bindValue(':key', $key, SQLITE3_TEXT);
			$stmt->bindValue(':value', $value, SQLITE3_TEXT);
			$stmt->execute();
		}

		$stmt->close();
	}


	/**
	 * Protect directory with index.php and .htaccess files.
	 *
	 * Creates .htaccess that denies access to .db files but allows .css files.
	 *
	 * @param string $dir Directory path.
	 */
	public static function protect_directory($dir) {
		// Create index.php - die() immediately
		$index = trailingslashit($dir) . 'index.php';
		if (!file_exists($index)) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
			file_put_contents($index, "<?php\n// Silence is golden.\ndie();\n");
		}

		// Create .htaccess - Allow CSS but deny database files
		$htaccess = trailingslashit($dir) . '.htaccess';
		$rules = "# Protect database files but allow CSS\n";
		$rules .= "Options -Indexes\n\n";
		
		// Deny access to .db files
		$rules .= "<FilesMatch \"\\.db$\">\n";
		$rules .= "    <IfModule mod_authz_core.c>\n";
		$rules .= "        Require all denied\n";
		$rules .= "    </IfModule>\n";
		$rules .= "    <IfModule !mod_authz_core.c>\n";
		$rules .= "        Deny from all\n";
		$rules .= "    </IfModule>\n";
		$rules .= "</FilesMatch>\n\n";
		
		// Allow CSS files explicitly
		$rules .= "<FilesMatch \"\\.css$\">\n";
		$rules .= "    <IfModule mod_authz_core.c>\n";
		$rules .= "        Require all granted\n";
		$rules .= "    </IfModule>\n";
		$rules .= "    <IfModule !mod_authz_core.c>\n";
		$rules .= "        Allow from all\n";
		$rules .= "    </IfModule>\n";
		$rules .= "</FilesMatch>\n";
		
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
		file_put_contents($htaccess, $rules);
	}
}
