<?php
/**
 * SMTP Mailer
 * 
 * Purpose: Send emails using custom SMTP configuration or wp_mail
 * Location: /includes/class-smtp-mailer.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_SMTP_Mailer {
    
    /**
     * Send email using SMTP or wp_mail based on form configuration
     * 
     * @param int    $form_id Form ID
     * @param string $to      Recipient email
     * @param string $subject Email subject
     * @param string $body    Email body (HTML)
     * @return bool Success
     */
    public static function send($form_id, $to, $subject, $body) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        
        // Load configuration from database
        $config = self::load_config_from_db($form_id);
        
        // Send email without debug
        return self::send_mail($to, $subject, $body, $config, false);
    }
    
    /**
     * Send test email with debug logging
     * 
     * @param array  $config  SMTP configuration array
     * @param string $to      Recipient email
     * @param string $subject Email subject
     * @param string $body    Email body (HTML)
     * @return array Response with success, debug_log, and error details
     */
    public static function send_test_email($config, $to, $subject, $body) {
        // Send email with debug enabled
        return self::send_mail($to, $subject, $body, $config, true);
    }
    
    /**
     * Load email configuration from database
     * 
     * @param int $form_id Form ID
     * @return array Configuration array
     */
    private static function load_config_from_db($form_id) {
        return [
            'smtp_enabled'    => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_enabled', $form_id, '0'),
            'smtp_host'       => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_host', $form_id, ''),
            'smtp_port'       => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_port', $form_id, '587'),
            'smtp_username'   => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_username', $form_id, ''),
            'smtp_password'   => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_password', $form_id, ''),
            'smtp_encryption' => CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_encryption', $form_id, 'tls'),
            'from_name'       => CMBSQF_DB_Settings::get('notifications__sender__from_name', $form_id, get_bloginfo('name')),
            'from_email'      => CMBSQF_DB_Settings::get('notifications__sender__from_email', $form_id, get_bloginfo('admin_email')),
        ];
    }
    
    /**
     * Core email sending logic
     * 
     * @param string $to      Recipient
     * @param string $subject Subject
     * @param string $body    Body (HTML)
     * @param array  $config  Configuration
     * @param bool   $debug   Enable debug logging
     * @return bool|array Bool for production, array for debug
     */
    private static function send_mail($to, $subject, $body, $config, $debug) {
        if ($config['smtp_enabled'] === '1') {
            return self::send_via_smtp($to, $subject, $body, $config, $debug);
        } else {
            return self::send_via_wp_mail($to, $subject, $body, $config, $debug);
        }
    }
    
    /**
     * Send email via WordPress wp_mail()
     * 
     * @param string $to      Recipient  
     * @param string $subject Subject
     * @param string $body    Body (HTML)
     * @param array  $config  Configuration
     * @param bool   $debug   Enable debug logging
     * @return bool|array
     */
    private static function send_via_wp_mail($to, $subject, $body, $config, $debug) {
        $debug_log = '';
        $wp_mail_error = null;
        
        // Helper for timestamp with milliseconds
        $get_timestamp = function() {
            $microtime = microtime(true);
            $milliseconds = sprintf("%03d", ($microtime - floor($microtime)) * 1000);
            return gmdate('H:i:s') . '.' . $milliseconds;
        };
        
        if ($debug) {
            // Hook to capture errors
            $error_callback = function ($wp_error) use (&$wp_mail_error) {
                $wp_mail_error = $wp_error;
            };
            add_action('wp_mail_failed', $error_callback);
            
            $debug_log .= '[' . $get_timestamp() . '] ' . "=== WordPress Mail (PHP mail) Mode ===\n";
            $debug_log .= '[' . $get_timestamp() . '] ' . "From Name: {$config['from_name']}\n";
            $debug_log .= '[' . $get_timestamp() . '] ' . "From Email: {$config['from_email']}\n";
            $debug_log .= '[' . $get_timestamp() . '] ' . "Recipient: {$to}\n";
            $debug_log .= '[' . $get_timestamp() . '] ' . "Subject: {$subject}\n";
        }
        
        // Set headers
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            sprintf('From: %s <%s>', $config['from_name'], $config['from_email']),
        ];
        
        if ($debug) {
             $debug_log .= '[' . $get_timestamp() . '] ' . "Headers:\n" . implode("\n", $headers) . "\n";
             $debug_log .= '[' . $get_timestamp() . '] ' . "Sending via wp_mail()...\n";
        }
        
        // Send email
        $result = wp_mail($to, $subject, $body, $headers);
        
        if ($debug) {
            remove_action('wp_mail_failed', $error_callback);
            
            if ($result) {
                $debug_log .= '[' . $get_timestamp() . '] ' . "Result: Success\n";
            } else {
                $debug_log .= '[' . $get_timestamp() . '] ' . "Result: Failed\n";
            }
            
            $response = [
                'success' => $result,
                'debug_log' => $debug_log,
            ];
            
            if (!$result && $wp_mail_error && is_wp_error($wp_mail_error)) {
                $error_msg = $wp_mail_error->get_error_message();
                $debug_log .= '[' . $get_timestamp() . '] ' . "Error: {$error_msg}\n";
                $response['debug_log'] = $debug_log; // Update log with error
                
                $response['error_details'] = [
                    'code' => $wp_mail_error->get_error_code(),
                    'message' => $error_msg,
                ];
            }
            
            return $response;
        }
        
        return $result;
    }
    
    /**
     * Send email via custom SMTP
     * 
     * @param string $to      Recipient
     * @param string $subject Subject
     * @param string $body    Body (HTML)
     * @param array  $config  Configuration
     * @param bool   $debug   Enable debug logging
     * @return bool|array
     */
    private static function send_via_smtp($to, $subject, $body, $config, $debug) {
        $debug_log = '';
        $wp_mail_error = null;
        
        // Helper for timestamp with milliseconds
        $get_timestamp = function() {
            $microtime = microtime(true);
            $milliseconds = sprintf("%03d", ($microtime - floor($microtime)) * 1000);
            return gmdate('H:i:s') . '.' . $milliseconds;
        };
        
        // Configure PHPMailer with SMTP
        $phpmailer_callback = function($phpmailer) use ($config, $debug, &$debug_log, $get_timestamp) {
            if ($debug) {
                // Enable SMTP debug output
                $phpmailer->SMTPDebug = 3;
                
                // Capture debug output
                $phpmailer->Debugoutput = function ($str) use (&$debug_log, $get_timestamp) {
                    $debug_log .= '[' . $get_timestamp() . '] ' . $str . "\n";
                };
                
                $debug_log .= '[' . $get_timestamp() . '] ' . "=== SMTP Mode Configuration ===\n";
                $debug_log .= '[' . $get_timestamp() . '] ' . "Host: {$config['smtp_host']}\n";
                $debug_log .= '[' . $get_timestamp() . '] ' . "Port: {$config['smtp_port']}\n";
                $debug_log .= '[' . $get_timestamp() . '] ' . "Encryption: {$config['smtp_encryption']}\n";
                $debug_log .= '[' . $get_timestamp() . '] ' . "Auth: " . (!empty($config['smtp_username']) ? 'Enabled' : 'Disabled') . "\n";
                $debug_log .= '[' . $get_timestamp() . '] ' . "============================\n\n";
            }
            
            $phpmailer->isSMTP();
            $phpmailer->Host = $config['smtp_host'];
            $phpmailer->Port = $config['smtp_port'];
            $phpmailer->SMTPAuth = !empty($config['smtp_username']);
            $phpmailer->Username = $config['smtp_username'];
            $phpmailer->Password = $config['smtp_password'];
            $phpmailer->SMTPSecure = $config['smtp_encryption'] === 'none' ? '' : $config['smtp_encryption'];
            $phpmailer->Timeout = 10;
        };
        
        if ($debug) {
            // Hook to capture errors
            $error_callback = function ($wp_error) use (&$wp_mail_error) {
                $wp_mail_error = $wp_error;
            };
            add_action('wp_mail_failed', $error_callback);
        }
        
        add_action('phpmailer_init', $phpmailer_callback);
        
        // Prepare headers
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            sprintf('From: %s <%s>', $config['from_name'], $config['from_email']),
        ];
        
        // Send via wp_mail
        $result = wp_mail($to, $subject, $body, $headers);
        
        // Remove hooks
        remove_action('phpmailer_init', $phpmailer_callback);
        
        if ($debug) {
            remove_action('wp_mail_failed', $error_callback);
            
            $response = [
                'success' => $result,
                'debug_log' => $debug_log,
            ];
            
            if (!$result && $wp_mail_error && is_wp_error($wp_mail_error)) {
                $response['error_details'] = [
                    'code' => $wp_mail_error->get_error_code(),
                    'message' => $wp_mail_error->get_error_message(),
                ];
            }
            
            return $response;
        }
        
        return $result;
    }
}
