<?php
/**
 * CSS Storage Helper
 *
 * Purpose: Manage storage and retrieval of custom CSS files.
 * Location: /includes/class-css-storage.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_CSS_Storage {

    /**
     * Get the file path for custom CSS.
     *
     * @param int $form_id Form ID (0 for global, > 0 for form-specific).
     * @return string|WP_Error Absolute path to the CSS file.
     */
    public static function get_file_path($form_id) {
        $form_id = intval($form_id);
        if ($form_id < 0) {
            return new WP_Error('invalid_form_id', __('Invalid form ID.', 'cmb-sqlite-form'));
        }

        $upload_dir = wp_upload_dir();
        if (!empty($upload_dir['error'])) {
            return new WP_Error('upload_dir_error', $upload_dir['error']);
        }

        $base_path = trailingslashit($upload_dir['basedir']) . 'cmb-sqlite-form/';
        
        // Global CSS: global/style.css
        // Form CSS: forms/{id}/style.css
        if ($form_id === 0) {
            $file_path = $base_path . 'global/style.css';
        } else {
            $file_path = $base_path . 'forms/' . $form_id . '/style.css';
        }
        
        return wp_normalize_path($file_path);
    }

    /**
     * Get the URL for custom CSS.
     *
     * @param int $form_id Form ID (0 for global, > 0 for form-specific).
     * @return string|WP_Error URL to the CSS file.
     */
    public static function get_file_url($form_id) {
        $form_id = intval($form_id);
        if ($form_id < 0) {
            return new WP_Error('invalid_form_id', __('Invalid form ID.', 'cmb-sqlite-form'));
        }

        $upload_dir = wp_upload_dir();
        if (!empty($upload_dir['error'])) {
            return new WP_Error('upload_dir_error', $upload_dir['error']);
        }

        $base_url = trailingslashit($upload_dir['baseurl']) . 'cmb-sqlite-form/';
        
        // Global CSS: global/style.css
        // Form CSS: forms/{id}/style.css
        if ($form_id === 0) {
            return $base_url . 'global/style.css';
        } else {
            return $base_url . 'forms/' . $form_id . '/style.css';
        }
    }

    /**
     * Save custom CSS to file.
     *
     * @param string $css     CSS content.
     * @param int    $form_id Form ID.
     * @return bool|WP_Error True on success, error otherwise.
     */
    public static function save($css, $form_id) {
        $file_path = self::get_file_path($form_id);
        if (is_wp_error($file_path)) {
            return $file_path;
        }

        // Ensure directory exists (it should, but safety first)
        $dir = dirname($file_path);
        if (!file_exists($dir)) {
            if (!wp_mkdir_p($dir)) {
                return new WP_Error('dir_create_failed', __('Failed to create form directory.', 'cmb-sqlite-form'));
            }
            // If we created it, protect it (though DB Core usually handles this)
            if (class_exists('CMBSQF_DB_Core_Installer')) {
                CMBSQF_DB_Core_Installer::protect_directory($dir);
            }
        }

        // Write file
        $result = file_put_contents($file_path, $css);

        return $result !== false;
    }

    /**
     * Get custom CSS content from file.
     *
     * @param int $form_id Form ID.
     * @return string CSS content or empty string if not found.
     */
    public static function get($form_id) {
        $file_path = self::get_file_path($form_id);
        if (is_wp_error($file_path) || !file_exists($file_path)) {
            return '';
        }

        return file_get_contents($file_path);
    }
}
