<?php

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Early context detection helper so the main plugin file can decide what to load.
 */
class CMBSQF_Context {
    /**
     * Determine the current request context.
     */
    public static function get() {
        if (wp_doing_ajax()) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only context detection
            $action = isset($_REQUEST['action']) ? sanitize_key(wp_unslash($_REQUEST['action'])) : '';

            // Detect plugin-specific AJAX actions
            if (strpos($action, CMBSQF_Constants::AJAX_PREFIX_ADMIN) === 0) {
                return 'ajax-admin';
            }

            if (strpos($action, CMBSQF_Constants::AJAX_PREFIX_FRONT) === 0) {
                return 'ajax-frontend';
            }

            return 'ajax-generic';
        }

        // Check for REST request (must check URL early, as REST_REQUEST is defined later)
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $is_rest = (strpos($request_uri, '/wp-json/') !== false) || (defined('REST_REQUEST') && REST_REQUEST);
        
        if ($is_rest) {
            $route = self::get_rest_route();

            // Check against plugin REST routes
            if (strpos($route, '/' . CMBSQF_Constants::REST_NAMESPACE . '/admin') === 0) {
                return 'rest-admin';
            }

            if (strpos($route, '/' . CMBSQF_Constants::REST_NAMESPACE . '/frontend') === 0) {
                return 'rest-frontend';
            }

            return 'rest-generic';
        }

        // Check for admin-post.php
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only context detection
        if (isset($_SERVER['SCRIPT_NAME']) && strpos(sanitize_text_field(wp_unslash($_SERVER['SCRIPT_NAME'])), 'admin-post.php') !== false) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only context detection
            $action = isset($_REQUEST['action']) ? sanitize_key(wp_unslash($_REQUEST['action'])) : '';
            
            // Check if action belongs to this plugin
            if (strpos($action, 'cmbsqf_') === 0) {
                return 'admin-post-plugin';
            }
            
            return 'admin-post';
        }

        if (is_admin()) {
            $page = self::get_admin_page_slug();
            if ($page && in_array($page, self::get_plugin_page_slugs(), true)) {
                return 'wp-admin-plugin';
            }

            return 'wp-admin';
        }

        return 'frontend';
    }

    /**
     * Normalize current admin page slug (if any).
     */
    private static function get_admin_page_slug() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only context detection
        if (!isset($_GET['page'])) {
            return '';
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only context detection
        return sanitize_key(wp_unslash($_GET['page']));
    }

    /**
     * List of slugs owned by this plugin.
     *
     * @return string[]
     */
    private static function get_plugin_page_slugs() {
        return [
            CMBSQF_Constants::SLUG_DASHBOARD,
            CMBSQF_Constants::SLUG_FORMS,
            CMBSQF_Constants::SLUG_SETTINGS,
        ];
    }

    /**
     * Resolve REST route whether using pretty permalinks or ?rest_route.
     *
     * @return string
     */
    private static function get_rest_route() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only context detection
        if (isset($_GET['rest_route'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only
            $route = sanitize_text_field(wp_unslash($_GET['rest_route']));
            return '/' . ltrim($route, '/');
        }

        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($request_uri === '') {
            return '';
        }

        $path   = wp_parse_url($request_uri, PHP_URL_PATH);
        $prefix = '/' . rest_get_url_prefix();

        if ($path && strpos($path, $prefix) !== false) {
            $route = substr($path, strpos($path, $prefix) + strlen($prefix));
            return '/' . ltrim($route, '/');
        }

        return '';
    }
}
