<?php
/**
 * Blocklist Manager
 *
 * Purpose: Manages disposable email domain blocklists
 * Location: /includes/class-blocklist-manager.php
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class CMBSQF_Blocklist_Manager
 *
 * Handles downloading, parsing, and validating email blocklists.
 */
class CMBSQF_Blocklist_Manager {

    /**
     * Update blocklist from multiple sources.
     *
     * @param int   $form_id Form ID (0 for global).
     * @param array $sources Array of URLs to download.
     * @return array Response with success status and metadata.
     */
    public static function update_blocklist($form_id, $sources) {
        $sources_metadata = [];
        $all_domains = [];

        // Download and parse each source
        foreach ($sources as $url) {
            $url = trim($url);
            if (empty($url)) {
                continue;
            }

            $metadata = [
                'url' => $url,
                'status' => 'pending',
                'domains' => 0,
                'last_updated' => current_time('mysql'),
                'error' => null,
            ];

            // Download source
            $content = self::download_source($url);

            if (is_wp_error($content)) {
                $metadata['status'] = 'error';
                $metadata['error'] = $content->get_error_message();
                $sources_metadata[] = $metadata;
                continue;
            }

            // Parse source
            $domains = self::parse_source($content, $url);

            if (is_wp_error($domains)) {
                $metadata['status'] = 'error';
                $metadata['error'] = $domains->get_error_message();
                $sources_metadata[] = $metadata;
                continue;
            }

            // Success
            $metadata['status'] = 'success';
            $metadata['domains'] = count($domains);
            $sources_metadata[] = $metadata;

            // Collect domains
            $all_domains = array_merge($all_domains, $domains);
        }

        // Deduplicate domains
        $unique_domains = self::deduplicate_domains($all_domains);

        // Save to file
        $file_result = self::save_blocklist_file($form_id, $unique_domains);

        if (is_wp_error($file_result)) {
            return [
                'success' => false,
                'message' => $file_result->get_error_message(),
            ];
        }

        // Save metadata to database
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';

        CMBSQF_DB_Settings::set('form__junk_email__sources_metadata', json_encode($sources_metadata), $form_id);
        CMBSQF_DB_Settings::set('form__junk_email__total_domains', (string) count($unique_domains), $form_id);
        CMBSQF_DB_Settings::set('form__junk_email__last_update', (string) time(), $form_id);

        return [
            'success' => true,
            'total_domains' => count($unique_domains),
            'sources_metadata' => $sources_metadata,
            'message' => sprintf(
                /* translators: %d: number of unique domains loaded into blocklist */
                __('Blocklist updated successfully. %d unique domains loaded.', 'cmb-sqlite-form'),
                count($unique_domains)
            ),
        ];
    }

    /**
     * Validate a single source and return domain count.
     *
     * @param string $url Source URL.
     * @return int|WP_Error Domain count or error.
     */
    public static function validate_source($url) {
        $content = self::download_source($url);
        if (is_wp_error($content)) {
            return $content;
        }

        $domains = self::parse_source($content, $url);
        if (is_wp_error($domains)) {
            return $domains;
        }

        return count($domains);
    }

    /**
     * Download content from URL.
     *
     * @param string $url URL to download.
     * @return string|WP_Error Content or error.
     */
    private static function download_source($url) {
        // Security: Use wp_safe_remote_get to prevent SSRF (blocks local/private IPs)
        $response = wp_safe_remote_get($url, [
            'timeout'             => 5,       // Reduced timeout to prevent DoS
            'redirection'         => 3,       // Limit redirects
            'httpversion'         => '1.1',
            'blocking'            => true,
            'limit_response_size' => 524288,  // Limit to 512KB to prevent Memory DoS
            'sslverify'           => true,
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);

        if ($code !== 200) {
            return new WP_Error(
                'download_failed',
                /* translators: %d: HTTP error code number (e.g., 404, 500) */
                sprintf(__('HTTP Error %d', 'cmb-sqlite-form'), $code)
            );
        }

        $body = wp_remote_retrieve_body($response);

        if (empty($body)) {
            return new WP_Error(
                'empty_response',
                __('Empty response from source', 'cmb-sqlite-form')
            );
        }

        // Double check content length if header exists (redundant with limit_response_size but good practice)
        $content_length = wp_remote_retrieve_header($response, 'content-length');
        if ($content_length && $content_length > 524288) {
            return new WP_Error(
                'file_too_large',
                __('File too large (max 512KB)', 'cmb-sqlite-form')
            );
        }

        return $body;
    }

    /**
     * Parse source content (txt or json).
     *
     * @param string $content Content to parse.
     * @param string $url     Source URL (for format detection).
     * @return array|WP_Error Array of domains or error.
     */
    private static function parse_source($content, $url) {
        $domains = [];

        // Detect format from URL
        if (strpos($url, '.json') !== false) {
            // JSON format
            $decoded = json_decode($content, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                return new WP_Error(
                    'json_parse_error',
                    __('Invalid JSON format', 'cmb-sqlite-form')
                );
            }

            // Handle different JSON structures
            if (is_array($decoded)) {
                // Flat array - validate each domain
                $domains = array_filter(array_map('trim', $decoded), [__CLASS__, 'is_valid_domain']);
            } else {
                return new WP_Error(
                    'json_structure_error',
                    __('Unsupported JSON structure', 'cmb-sqlite-form')
                );
            }
        } else {
            // Plain text format (one domain per line)
            $lines = explode("\n", $content);
            foreach ($lines as $line) {
                $line = trim($line);
                
                // Skip empty lines and comments
                if (empty($line) || strpos($line, '#') === 0) {
                    continue;
                }
                
                // Validate domain before adding
                if (self::is_valid_domain($line)) {
                    $domains[] = $line;
                }
            }
        }

        // Ensure we have at least some valid domains
        $valid_domains = array_values(array_filter($domains));
        
        if (empty($valid_domains)) {
            return new WP_Error(
                'no_valid_domains',
                __('No valid domains found in source', 'cmb-sqlite-form')
            );
        }

        return $valid_domains;
    }

    /**
     * Validate if a string is a valid domain.
     *
     * @param string $domain Domain to validate.
     * @return bool True if valid domain.
     */
    private static function is_valid_domain($domain) {
        // Use filter_var with temporary scheme to validate domain format
        // Automatically rejects spaces, commas, HTML, and other invalid characters
        return filter_var('http://' . $domain, FILTER_VALIDATE_URL) !== false;
    }

    /**
     * Deduplicate domains.
     *
     * @param array $domains Array of domains.
     * @return array Unique domains.
     */
    private static function deduplicate_domains($domains) {
        // Convert to lowercase and deduplicate
        $domains = array_map('strtolower', $domains);
        $domains = array_unique($domains);
        $domains = array_values($domains); // Re-index
        
        return $domains;
    }

    /**
     * Save blocklist to file.
     *
     * @param int   $form_id Form ID (0 for global).
     * @param array $domains Array of domains.
     * @return true|WP_Error True on success, error on failure.
     */
    private static function save_blocklist_file($form_id, $domains) {
        $file_path = self::get_blocklist_path($form_id);
        $dir = dirname($file_path);

        // Create directory if it doesn't exist
        if (!file_exists($dir)) {
            if (!wp_mkdir_p($dir)) {
                return new WP_Error(
                    'dir_creation_failed',
                    __('Failed to create directory', 'cmb-sqlite-form')
                );
            }
        }

        // Write domains to file (one per line)
        $content = implode("\n", $domains);
        $result = file_put_contents($file_path, $content);

        if ($result === false) {
            return new WP_Error(
                'file_write_failed',
                __('Failed to write blocklist file', 'cmb-sqlite-form')
            );
        }

        return true;
    }

    /**
     * Get blocklist file path.
     *
     * @param int $form_id Form ID (0 for global).
     * @return string File path.
     */
    public static function get_blocklist_path($form_id) {
        $upload_dir = wp_upload_dir();
        $base_dir = trailingslashit($upload_dir['basedir']) . 'cmb-sqlite-form';

        if ($form_id === 0) {
            return $base_dir . '/global/junk-domains.txt';
        } else {
            return $base_dir . '/forms/' . $form_id . '/junk-domains.txt';
        }
    }

    /**
     * Check if email domain is disposable.
     *
     * @param string $email   Email address.
     * @param int    $form_id Form ID.
     * @return bool True if disposable.
     */
    public static function is_disposable_email($email, $form_id) {
        // Extract domain
        $parts = explode('@', $email);
        if (count($parts) !== 2) {
            return false;
        }

        $domain = strtolower(trim($parts[1]));

        // Get blocklist file path
        $file_path = self::get_blocklist_path($form_id);

        // If file doesn't exist, not disposable
        if (!file_exists($file_path)) {
            return false;
        }

        // Load and cache blocklist
        static $cache = [];

        if (!isset($cache[$file_path])) {
            $content = file_get_contents($file_path);
            
            if ($content === false) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional error logging for admin visibility
                error_log('CMBSQF Error: Failed to read blocklist file at ' . $file_path);
                return false;
            }

            $domains = explode("\n", $content);
            $domains = array_map('trim', $domains);
            $domains = array_filter($domains); // Remove empty lines
            $cache[$file_path] = array_flip($domains); // Use array_flip for O(1) lookup
        }

        return isset($cache[$file_path][$domain]);
    }
}
