<?php
/**
 * Form Validator
 * 
 * Purpose: Server-side validation of form submissions
 * Location: /frontend/class-form-validator.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_Form_Validator {
    
    /**
     * Validate submission data
     * 
     * @param int   $form_id Form ID
     * @param array $data    Submission data
     * @return array Array of error messages (empty if valid)
     */
    public static function validate($form_id, $data) {
        $errors = [];
        
        // Load field configuration
        $fields_config = self::load_fields_config($form_id);
        
        // Validate required fields
        foreach ($fields_config as $field_name => $config) {
            if ($config['required'] && empty($data[$field_name])) {
                $errors[] = sprintf(
                    /* translators: %s: field name (e.g., "Name", "Email") */
                    __('%s is required.', 'cmb-sqlite-form'),
                    ucfirst($field_name)
                );
            }
        }
        
        // Validate email format
        if (!empty($data['email']) && !is_email($data['email'])) {
            $errors[] = __('Invalid email address.', 'cmb-sqlite-form');
        }
        
        // Validate URL format (if provided)
        if (!empty($data['website']) && !filter_var($data['website'], FILTER_VALIDATE_URL)) {
            $errors[] = __('Invalid website URL.', 'cmb-sqlite-form');
        }
        
        // Check blocklist for disposable emails
        $block_disposable = CMBSQF_DB_Settings::get('form__junk_email__block_disposable', $form_id, '0');
        if ($block_disposable === '1' && !empty($data['email'])) {
            $is_blocked = self::check_email_blocklist($form_id, $data['email']);
            if ($is_blocked) {
                $message = CMBSQF_DB_Settings::get('form__junk_email__disposable_message', $form_id, '');
                // Use default message if custom message is empty
                if (empty($message)) {
                    $message = __('Disposable or temporary email addresses are not allowed. Please use a permanent email address.', 'cmb-sqlite-form');
                }
                $errors[] = $message;
            }
        }
        
        // Privacy validation
        $privacy_enabled = CMBSQF_DB_Settings::get('form__privacy__enabled', $form_id, '0');
        if ($privacy_enabled === '1' && $data['privacy_status'] !== 'accepted') {
            $errors[] = __('You must accept the privacy policy.', 'cmb-sqlite-form');
        }
        
        // Bot Protection: Honeypot validation
        $honeypot_enabled = CMBSQF_DB_Settings::get('form__bot_protection__honeypot', $form_id, '1');
        if ($honeypot_enabled === '1') {
            $honeypot_field = CMBSQF_DB_Settings::get('form__bot_protection__honeypot_field', $form_id, 'business_url');
            if (!empty($data[$honeypot_field])) {
                // Honeypot was filled - this is a bot
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    $errors[] = sprintf('[DEBUG] Honeypot validation failed. Field "%s" contains: "%s"', $honeypot_field, $data[$honeypot_field]);
                } else {
                    $errors[] = __('Your submission could not be processed. Please try again.', 'cmb-sqlite-form');
                }
            }
        }
        
        // Bot Protection: Min time validation
        $min_time_enabled = CMBSQF_DB_Settings::get('form__bot_protection__min_time_enabled', $form_id, '1');
        if ($min_time_enabled === '1' && isset($data['form_loaded_at'])) {
            $min_time = (int) CMBSQF_DB_Settings::get('form__bot_protection__min_time', $form_id, '3');
            $loaded_at = (int) $data['form_loaded_at'];
            $current_time = time();
            $elapsed = $current_time - $loaded_at;
            
            if ($elapsed < $min_time) {
                $errors[] = __('Form submitted too quickly. Please wait a moment and try again.', 'cmb-sqlite-form');
            }
        }
        
        // Bot Protection: Captcha validation
        $captcha_enabled = CMBSQF_DB_Settings::get('form__bot_protection__captcha_enabled', $form_id, '0');
        if ($captcha_enabled === '1') {
            $captcha_answer = CMBSQF_DB_Settings::get('form__bot_protection__captcha_answer', $form_id, '');
            $user_response = isset($data['captcha_response']) ? trim($data['captcha_response']) : '';
            
            // Case-insensitive comparison
            if (empty($captcha_answer) || strcasecmp($user_response, trim($captcha_answer)) !== 0) {
                $errors[] = __('Incorrect answer to the verification question.', 'cmb-sqlite-form');
            }
        }
        
        return $errors;
    }
    
    /**
     * Load field configuration from settings
     * 
     * @param int $form_id Form ID
     * @return array Field configuration
     */
    private static function load_fields_config($form_id) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        
        $fields = CMBSQF_Constants::FORM_FIELDS;
        $fields_config = [];
        
        foreach ($fields as $field) {
            $enabled = CMBSQF_DB_Settings::get("form__fields__{$field}_enabled", $form_id, '0');
            
            if ($enabled === '1') {
                $fields_config[$field] = [
                    'enabled'  => true,
                    'required' => CMBSQF_DB_Settings::get("form__fields__{$field}_required", $form_id, '0') === '1',
                ];
            }
        }
        
        return $fields_config;
    }
    
    /**
     * Check if email domain is in blocklist
     * 
     * @param int    $form_id Form ID
     * @param string $email   Email address
     * @return bool True if blocked
     */
    private static function check_email_blocklist($form_id, $email) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-blocklist-manager.php';
        
        return CMBSQF_Blocklist_Manager::is_disposable_email($email, $form_id);
    }
}
