<?php
/**
 * Form Renderer
 *
 * Purpose: Renders frontend forms with dynamic fields
 * Location: /frontend/class-form-renderer.php
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class CMBSQF_Form_Renderer
 *
 * Handles rendering of forms on the frontend.
 */
class CMBSQF_Form_Renderer {

    /**
     * Render a form by ID.
     *
     * @param int $form_id Form ID.
     * @return string HTML output.
     */
    public static function render($form_id) {
        $form_id = intval($form_id);

        if ($form_id <= 0) {
            return '<div class="cmbsqf-error">' . esc_html__('Invalid form ID', 'cmb-sqlite-form') . '</div>';
        }

        // Load settings
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        
        $fields_config = self::get_fields_config($form_id);
        $ui_texts = self::get_ui_texts($form_id);
        
        // Enqueue assets
        self::enqueue_assets($form_id);

        // Render form HTML
        ob_start();
        self::render_form_html($form_id, $fields_config, $ui_texts);
        return ob_get_clean();
    }

    /**
     * Get field configuration for form.
     *
     * @param int $form_id Form ID.
     * @return array Fields configuration.
     */
    private static function get_fields_config($form_id) {
        $fields = CMBSQF_Constants::FORM_FIELDS;
        $config = [];

        foreach ($fields as $field) {
            $enabled = CMBSQF_DB_Settings::get("form__fields__{$field}_enabled", $form_id, '0');
            
            if ($enabled === '1') {
                $config[$field] = [
                    'enabled'  => true,
                    'required' => CMBSQF_DB_Settings::get("form__fields__{$field}_required", $form_id, '0') === '1',
                    'label'    => CMBSQF_DB_Settings::get("form__fields__{$field}_label", $form_id, ucfirst($field)),
                ];
            }
        }

        return $config;
    }

    /**
     * Get UI texts for form.
     *
     * @param int $form_id Form ID.
     * @return array UI texts.
     */
    private static function get_ui_texts($form_id) {
        // Get privacy settings with proper fallback
        $privacy_enabled = CMBSQF_DB_Settings::get('form__privacy__enabled', $form_id, '0');
        $privacy_html = CMBSQF_DB_Settings::get('form__privacy__html', $form_id, '');
        
        // If privacy is enabled but text is empty, fallback to global
        if ($privacy_enabled === '1' && empty($privacy_html)) {
            $privacy_html = CMBSQF_DB_Settings::get_global_setting('form__privacy__html', '');
        }
        
        // Get bot protection settings
        $honeypot_enabled = CMBSQF_DB_Settings::get('form__bot_protection__honeypot', $form_id, '1');
        $honeypot_field = CMBSQF_DB_Settings::get('form__bot_protection__honeypot_field', $form_id, 'business_url');
        $min_time_enabled = CMBSQF_DB_Settings::get('form__bot_protection__min_time_enabled', $form_id, '1');
        $min_time = CMBSQF_DB_Settings::get('form__bot_protection__min_time', $form_id, '3');
        $captcha_enabled = CMBSQF_DB_Settings::get('form__bot_protection__captcha_enabled', $form_id, '0');
        $captcha_question = CMBSQF_DB_Settings::get('form__bot_protection__captcha_question', $form_id, '');
        
        return [
            'title'           => CMBSQF_DB_Settings::get('form__ui_texts__title', $form_id, ''),
            'header_html'     => CMBSQF_DB_Settings::get('form__ui_texts__header_html', $form_id, ''),
            'submit_button'   => CMBSQF_DB_Settings::get('form__ui_texts__submit_button', $form_id, __('Submit', 'cmb-sqlite-form')),
            'privacy_enabled' => $privacy_enabled,
            'privacy_html'    => $privacy_html,
            // Bot protection
            'honeypot_enabled' => $honeypot_enabled,
            'honeypot_field'   => $honeypot_field,
            'min_time_enabled' => $min_time_enabled,
            'min_time'         => $min_time,
            'captcha_enabled'  => $captcha_enabled,
            'captcha_question' => $captcha_question,
        ];
    }

    /**
     * Enqueue CSS and JS assets.
     *
     * @param int $form_id Form ID.
     */
    private static function enqueue_assets($form_id) {
        // Enqueue custom CSS for this form
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-css-storage.php';
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        
        // Check if styling override is enabled
        $override = CMBSQF_DB_Settings::get_form_setting('form__styling__override', $form_id, '0');
        
        // If override is disabled, use global CSS (form_id = 0)
        $css_form_id = ($override === '1') ? $form_id : 0;
        
        $css_path = CMBSQF_CSS_Storage::get_file_path($css_form_id);
        $css_url = CMBSQF_CSS_Storage::get_file_url($css_form_id);
        
        if (!is_wp_error($css_path) && file_exists($css_path)) {
            wp_enqueue_style(
                'cmbsqf-form-' . $form_id,
                $css_url,
                [],
                filemtime($css_path)
            );
        }

        // Enqueue frontend JavaScript
        wp_enqueue_script(
            'cmbsqf-form',
            plugin_dir_url(__FILE__) . 'assets/js/form.js',
            [],
            CMBSQF_VERSION,
            true
        );

        wp_localize_script('cmbsqf-form', 'cmbsqfData', [
            'restUrl' => rest_url('cmbsqf/v1/frontend/submit'),
            'nonce'   => wp_create_nonce('wp_rest'),
            'i18n'    => [
                'minTimeError'  => __('Please wait a moment before submitting the form.', 'cmb-sqlite-form'),
                'submitting'    => __('Sending...', 'cmb-sqlite-form'),
            ],
        ]);
    }

    /**
     * Render form HTML.
     *
     * @param int   $form_id       Form ID.
     * @param array $fields_config Field configuration.
     * @param array $ui_texts      UI texts.
     */
    private static function render_form_html($form_id, $fields_config, $ui_texts) {
        ?>
        <div class="cmbsqf-wrap">
            <?php if (!empty($ui_texts['title'])): ?>
                <h2 class="cmbsqf-title"><?php echo esc_html($ui_texts['title']); ?></h2>
            <?php endif; ?>

            <?php if (!empty($ui_texts['header_html'])): ?>
                <div class="cmbsqf-header">
                    <?php echo wp_kses_post($ui_texts['header_html']); ?>
                </div>
            <?php endif; ?>

            <form class="cmbsqf-form" data-form-id="<?php echo esc_attr($form_id); ?>">
                <?php foreach ($fields_config as $field_name => $config): ?>
                    <?php self::render_field($field_name, $config); ?>
                <?php endforeach; ?>

                <?php 
                // Bot Protection: Honeypot field (hidden)
                if ($ui_texts['honeypot_enabled'] === '1'): 
                ?>
                    <div style="position:absolute;left:-9999px;opacity:0;pointer-events:none;" aria-hidden="true" tabindex="-1">
                        <label for="<?php echo esc_attr($ui_texts['honeypot_field']); ?>"><?php esc_html_e('Leave this field empty', 'cmb-sqlite-form'); ?></label>
                        <input 
                            type="text"
                            id="<?php echo esc_attr($ui_texts['honeypot_field']); ?>"
                            name="<?php echo esc_attr($ui_texts['honeypot_field']); ?>"
                            value=""
                            autocomplete="off"
                            tabindex="-1"
                        />
                    </div>
                <?php endif; ?>

                <?php 
                // Bot Protection: Timestamp for min_time validation
                if ($ui_texts['min_time_enabled'] === '1'): 
                ?>
                    <input 
                        type="hidden" 
                        name="form_loaded_at" 
                        value="<?php echo esc_attr(time()); ?>"
                        data-min-time="<?php echo esc_attr($ui_texts['min_time']); ?>"
                    />
                <?php endif; ?>

                <?php 
                // Bot Protection: Captcha question
                if ($ui_texts['captcha_enabled'] === '1' && !empty($ui_texts['captcha_question'])): 
                ?>
                    <div class="cmbsqf-field cmbsqf-field--captcha">
                        <label for="cmbsqf-captcha" class="cmbsqf-label">
                            <?php echo esc_html($ui_texts['captcha_question']); ?> <span class="cmbsqf-required">*</span>
                        </label>
                        <input 
                            type="text"
                            id="cmbsqf-captcha"
                            name="captcha_response"
                            class="cmbsqf-input"
                            required
                            autocomplete="off"
                        />
                    </div>
                <?php endif; ?>

                <?php if ($ui_texts['privacy_enabled'] === '1'): ?>
                    <div class="cmbsqf-field cmbsqf-field--privacy">
                        <label class="cmbsqf-privacy-label">
                            <input 
                                type="checkbox"
                                id="cmbsqf-privacy"
                                name="privacy_status"
                                value="accepted"
                                class="cmbsqf-privacy-checkbox"
                                required
                            />
                            <?php if (!empty($ui_texts['privacy_html'])): ?>
                                <span class="cmbsqf-privacy-text"><?php echo wp_kses_post($ui_texts['privacy_html']); ?></span>
                            <?php else: ?>
                                <span class="cmbsqf-privacy-text"><?php esc_html_e('I accept the privacy policy', 'cmb-sqlite-form'); ?></span>
                            <?php endif; ?>
                        </label>
                    </div>
                <?php endif; ?>

                <div class="cmbsqf-submit">
                    <button type="submit" class="cmbsqf-button">
                        <?php echo esc_html($ui_texts['submit_button']); ?>
                    </button>
                </div>

                <div class="cmbsqf-message" style="display: none;"></div>
            </form>
        </div>
        <?php
    }

    /**
     * Render individual field.
     *
     * @param string $field_name Field name.
     * @param array  $config     Field configuration.
     */
    private static function render_field($field_name, $config) {
        $label = $config['label'];
        $required = $config['required'];
        $required_attr = $required ? 'required' : '';
        $required_mark = $required ? ' <span class="cmbsqf-required">*</span>' : '';

        ?>
        <div class="cmbsqf-field cmbsqf-field--<?php echo esc_attr($field_name); ?>">
            <label for="cmbsqf-<?php echo esc_attr($field_name); ?>" class="cmbsqf-label">
                <?php echo esc_html($label); ?><?php echo wp_kses_post($required_mark); ?>
            </label>

            <?php if ($field_name === 'message'): ?>
                <textarea 
                    id="cmbsqf-<?php echo esc_attr($field_name); ?>"
                    name="<?php echo esc_attr($field_name); ?>"
                    class="cmbsqf-textarea"
                    rows="5"
                    <?php echo esc_attr($required_attr); ?>
                ></textarea>
            <?php else: ?>
                <?php
                $type = ($field_name === 'email') ? 'email' : 'text';
                if ($field_name === 'phone') {
                    $type = 'tel';
                } elseif ($field_name === 'website') {
                    $type = 'url';
                }
                ?>
                <input 
                    type="<?php echo esc_attr($type); ?>"
                    id="cmbsqf-<?php echo esc_attr($field_name); ?>"
                    name="<?php echo esc_attr($field_name); ?>"
                    class="cmbsqf-input"
                    <?php echo esc_attr($required_attr); ?>
                />
            <?php endif; ?>
        </div>
        <?php
    }
}
