<?php
/**
 * Form Handler
 * 
 * Purpose: Processes validated form submissions
 * Location: /frontend/class-form-handler.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_Form_Handler {
    
    /**
     * Process form submission: validate, save to database, and send email notifications.
     *
     * @param int   $form_id Form ID.
     * @param array $data    Validated submission data (name, email, message, etc.).
     * @return array|WP_Error Array with 'success' and 'email_sent' keys on success, WP_Error on failure.
     */
    public static function process($form_id, $data) {
        // Save to database
        $submission_id = self::save_submission($form_id, $data);
        
        if (!$submission_id) {
            return new WP_Error('save_failed', __('Failed to save submission.', 'cmb-sqlite-form'));
        }
        
        // Add submission_id to data for email placeholders
        $data['submission_id'] = $submission_id;
        
        // Send admin notification email
        self::send_admin_notification($form_id, $data);
        
        // Send user notification if enabled and capture result
        $email_sent = self::send_user_notification($form_id, $data);
        
        return [
            'success' => true,
            'email_sent' => $email_sent,
        ];
    }
    
    /**
     * Save submission to database
     * 
     * @param int   $form_id Form ID
     * @param array $data    Submission data
     * @return int|false Submission ID or false on failure
     */
    private static function save_submission($form_id, $data) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        
        $submission_data = [
            'form_id'      => $form_id,
            'name'         => $data['name'] ?? '',
            'surname'      => $data['surname'] ?? '',
            'email'        => $data['email'] ?? '',
            'phone'        => $data['phone'] ?? '',
            'organization' => $data['organization'] ?? '',
            'website'      => $data['website'] ?? '',
            'position'     => $data['position'] ?? '',
            'subject'      => $data['subject'] ?? '',
            'message'      => $data['message'] ?? '',
            'privacy_status' => $data['privacy_status'] ?? 'not_configured',
            'submitted_at' => current_time('mysql'),
        ];
        
        return CMBSQF_DB_Submissions::create($submission_data);
    }
    
    /**
     * Send admin notification email
     * 
     * @param int   $form_id Form ID
     * @param array $data    Submission data
     */
    private static function send_admin_notification($form_id, $data) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-smtp-mailer.php';
        
        // Check if notifications are enabled
        $enabled = CMBSQF_DB_Settings::get('notifications__to_admin__enabled', $form_id, '1');
        if ($enabled !== '1') {
            return;
        }
        
        // Get admin email(s)
        $admin_emails = CMBSQF_DB_Settings::get('notifications__to_admin__emails', $form_id, get_option('admin_email'));
        
        // Get email subject and message
        $subject = CMBSQF_DB_Settings::get('notifications__to_admin__subject', $form_id, 'New form submission from {name}');
        $message = CMBSQF_DB_Settings::get('notifications__to_admin__message', $form_id, '');
        
        // If custom message not set, use build_notification_body
        if (empty($message)) {
            $body = self::build_notification_body($data);
        } else {
            // Replace placeholders in custom message
            $body = self::replace_placeholders($form_id, $message, $data);
            $body = wpautop($body); // Convert newlines to <p> and <br>
        }
        
        // Send email via SMTP Mailer (don't track admin emails)
        CMBSQF_SMTP_Mailer::send($form_id, $admin_emails, self::replace_placeholders($form_id, $subject, $data), $body);
    }
    
    /**
     * Build notification email body
     * 
     * @param array $data Submission data
     * @return string Email body HTML
     */
    private static function build_notification_body($data) {
        $body = '<h2>' . __('New Form Submission', 'cmb-sqlite-form') . '</h2>';
        $body .= '<table style="border-collapse: collapse; width: 100%;">';
        
        $fields = [
            'name'         => __('Name', 'cmb-sqlite-form'),
            'surname'      => __('Surname', 'cmb-sqlite-form'),
            'email'        => __('Email', 'cmb-sqlite-form'),
            'phone'        => __('Phone', 'cmb-sqlite-form'),
            'organization' => __('Organization', 'cmb-sqlite-form'),
            'website'      => __('Website', 'cmb-sqlite-form'),
            'position'     => __('Position', 'cmb-sqlite-form'),
            'subject'      => __('Subject', 'cmb-sqlite-form'),
            'message'      => __('Message', 'cmb-sqlite-form'),
        ];
        
        foreach ($fields as $key => $label) {
            if (!empty($data[$key])) {
                $body .= '<tr>';
                $body .= '<td style="padding: 8px; border: 1px solid #ddd; font-weight: bold;">' . esc_html($label) . '</td>';
                $body .= '<td style="padding: 8px; border: 1px solid #ddd;">' . nl2br(esc_html($data[$key])) . '</td>';
                $body .= '</tr>';
            }
        }
        
        $body .= '</table>';
        
        return $body;
    }
    
    /**
     * Send user notification email to submitter
     * 
     * @param int   $form_id Form ID
     * @param array $data    Submission data
     * @return bool True if email sent successfully, false otherwise
     */
    private static function send_user_notification($form_id, $data) {
        if (empty($data['email'])) {
            return false;
        }
        
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-smtp-mailer.php';
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        
        // Check if user notification is enabled
        $enabled = CMBSQF_DB_Settings::get('notifications__to_user__enabled', $form_id, '0');
        if ($enabled !== '1') {
            // Update DB: user notification not configured
            CMBSQF_DB_Submissions::update_email_delivery(
                $form_id,
                $data['submission_id'],
                'not_configured',
                'User notification disabled'
            );
            return false;
        }
        
        // Get user notification content
        $subject = CMBSQF_DB_Settings::get('notifications__to_user__subject', $form_id, 'Thank you for contacting us');
        $body = CMBSQF_DB_Settings::get('notifications__to_user__message', $form_id, '');
        
        if (empty($body)) {
            // Update DB: no message configured
            CMBSQF_DB_Submissions::update_email_delivery(
                $form_id,
                $data['submission_id'],
                'not_configured',
                'User notification message not set'
            );
            return false;
        }
        
        // Replace placeholders
        $body = self::replace_placeholders($form_id, $body, $data);
        
        // Send email via SMTP Mailer and capture result
        $result = CMBSQF_SMTP_Mailer::send($form_id, $data['email'], self::replace_placeholders($form_id, $subject, $data), wpautop($body));
        
        // Update DB with delivery status
        CMBSQF_DB_Submissions::update_email_delivery(
            $form_id,
            $data['submission_id'],
            $result ? 'sent' : 'failed',
            $result ? null : 'User notification sending failed'
        );
        
        return $result;
    }
    
    /**
     * Replace placeholders in email content
     * 
     * @param int    $form_id Form ID
     * @param string $content Email subject or body
     * @param array  $data    Submission data
     * @return string Content with replaced placeholders
     */
    private static function replace_placeholders($form_id, $content, $data) {
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        
        // Get form name
        $form_name = CMBSQF_DB_Settings::get('form_name', $form_id, 'Form #' . $form_id);
        
        // Get IP address (from submission or current)
        $ip_address = $data['ip_address'] ?? (isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '');
        
        $placeholders = [
            // Form fields
            '{name}'         => $data['name'] ?? '',
            '{surname}'      => $data['surname'] ?? '',
            '{email}'        => $data['email'] ?? '',
            '{phone}'        => $data['phone'] ?? '',
            '{organization}' => $data['organization'] ?? '',
            '{website}'      => $data['website'] ?? '',
            '{position}'     => $data['position'] ?? '',
            '{subject}'      => $data['subject'] ?? '',
            '{message}'      => $data['message'] ?? '',
            
            // Meta fields
            '{form_name}'    => $form_name,
            '{site_name}'    => get_bloginfo('name'),
            '{site_url}'     => get_bloginfo('url'),
            '{ip_address}'   => $ip_address,
        ];
        
        return str_replace(array_keys($placeholders), array_values($placeholders), $content);
    }
}
