/**
 * Contact Form Frontend JavaScript
 * 
 * Handles form submission, validation, and AJAX communication
 */

(function () {
    'use strict';

    /**
     * Initialize form when DOM is ready
     */
    document.addEventListener('DOMContentLoaded', function () {
        const forms = document.querySelectorAll('.cmbsqf-form');

        forms.forEach(function (form) {
            new ContactForm(form);
        });
    });

    /**
     * ContactForm class
     */
    function ContactForm(formElement) {
        this.form = formElement;
        this.button = formElement.querySelector('.cmbsqf-button');
        this.messageContainer = formElement.querySelector('.cmbsqf-message');
        this.formId = formElement.dataset.formId;

        this.init();
    }

    ContactForm.prototype = {
        /**
         * Initialize form event listeners
         */
        init: function () {
            this.form.addEventListener('submit', this.handleSubmit.bind(this));
        },

        /**
         * Handle form submission
         */
        handleSubmit: function (e) {
            e.preventDefault();

            // Clear previous messages
            this.hideMessage();

            // Bot Protection: Check min_time
            const timestampField = this.form.querySelector('input[name="form_loaded_at"]');
            if (timestampField) {
                const loadedAt = parseInt(timestampField.value);
                const minTime = parseInt(timestampField.dataset.minTime || 0);
                const currentTime = Math.floor(Date.now() / 1000);
                const elapsed = currentTime - loadedAt;

                if (elapsed < minTime) {
                    this.showMessage(cmbsqfData.i18n.minTimeError, 'error');
                    return;
                }
            }

            // Validate form
            if (!this.validateForm()) {
                return;
            }

            // Collect form data
            const formData = this.collectFormData();

            // Submit via AJAX
            this.submitForm(formData);
        },

        /**
         * Validate form fields
         */
        validateForm: function () {
            const inputs = this.form.querySelectorAll('input[required], textarea[required], input[type="email"]');
            let isValid = true;

            inputs.forEach(function (input) {
                // Check required fields
                if (input.hasAttribute('required') && !input.value.trim()) {
                    isValid = false;
                    input.classList.add('cmbsqf-error');
                    return;
                }

                // Validate email format
                if (input.type === 'email' && input.value.trim()) {
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (!emailRegex.test(input.value)) {
                        isValid = false;
                        input.classList.add('cmbsqf-error');
                        return;
                    }
                }

                // Validate URL format
                if (input.type === 'url' && input.value.trim()) {
                    try {
                        new URL(input.value);
                    } catch (_) {
                        isValid = false;
                        input.classList.add('cmbsqf-error');
                        return;
                    }
                }

                // Remove error class if valid
                input.classList.remove('cmbsqf-error');
            }.bind(this));

            if (!isValid) {
                this.showMessage('Please fill in all required fields correctly.', 'error');
            }

            return isValid;
        },

        /**
         * Collect form data
         */
        collectFormData: function () {
            const formData = new FormData(this.form);
            const data = {
                form_id: this.formId
            };

            formData.forEach(function (value, key) {
                data[key] = value;
            });

            return data;
        },

        /**
         * Submit form via AJAX
         */
        submitForm: function (data) {
            // Disable button and show loading state
            this.setLoadingState(true);

            fetch(cmbsqfData.restUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': cmbsqfData.nonce
                },
                body: JSON.stringify(data)
            })
                .then(function (response) {
                    return response.json();
                })
                .then(function (result) {
                    // Response format: { success: true/false, data: { message: "..." } }
                    if (result.success) {
                        this.showMessage(result.data.message || 'Form submitted successfully!', 'success');
                        this.form.reset();
                    } else {
                        this.showMessage(result.data.message || 'An error occurred. Please try again.', 'error');
                    }
                }.bind(this))
                .catch(function (error) {
                    this.showMessage(error.message || 'An error occurred. Please try again.', 'error');
                }.bind(this))
                .finally(function () {
                    this.setLoadingState(false);
                }.bind(this));
        },

        /**
         * Set loading state
         */
        setLoadingState: function (isLoading) {
            if (isLoading) {
                this.button.disabled = true;
                this.button.dataset.originalText = this.button.textContent;
                this.button.textContent = cmbsqfData.i18n.submitting;
            } else {
                this.button.disabled = false;
                this.button.textContent = this.button.dataset.originalText || 'Submit';
            }
        },

        /**
         * Show message
         */
        showMessage: function (message, type) {
            this.messageContainer.textContent = message;
            this.messageContainer.className = 'cmbsqf-message cmbsqf-' + type;
            this.messageContainer.style.display = 'block';

            // Auto-hide success messages after 5 seconds
            if (type === 'success') {
                setTimeout(function () {
                    this.hideMessage();
                }.bind(this), 5000);
            }
        },

        /**
         * Hide message
         */
        hideMessage: function () {
            this.messageContainer.style.display = 'none';
            this.messageContainer.textContent = '';
        }
    };
})();
