<?php
/**
 * General Tab Renderer
 *
 * Purpose: Renders the General tab with Behavior, Data Management, and Info sections
 * Used by: Settings pages (global context)
 * Location: /admin/tabs/class-tab-general.php
 *
 * This tab uses the Sidebar Renderer to render a sidebar navigation with 3 sections.
 * All configuration is visible in this file - no need to check other files.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load Sidebar Renderer


// Load section classes
require_once CMBSQF_PLUGIN_DIR . 'admin/sections/class-section-behavior.php';
require_once CMBSQF_PLUGIN_DIR . 'admin/sections/class-section-data.php';
require_once CMBSQF_PLUGIN_DIR . 'admin/sections/class-section-info.php';

/**
 * Class CMBSQF_Tab_General
 *
 * Renders General tab with sidebar navigation for Behavior, Data, and Info sections.
 */
class CMBSQF_Tab_General {

    /**
     * Render the General tab.
     *
     * @param int $form_id Form ID (0 for global settings, specific ID for form settings).
     */
    public static function render($form_id = 0) {
        // Define sections for this tab
        $sections = [
            'behavior' => [
                'label'    => __('Behavior', 'cmb-sqlite-form'),
                'icon'     => self::get_icon('behavior'),
                'callback' => ['CMBSQF_Section_Behavior', 'render'],
            ],
            'data' => [
                'label'    => __('Data Management', 'cmb-sqlite-form'),
                'icon'     => self::get_icon('data'),
                'callback' => ['CMBSQF_Section_Data', 'render'],
            ],
            'info' => [
                'label'    => __('Information', 'cmb-sqlite-form'),
                'icon'     => self::get_icon('info'),
                'callback' => ['CMBSQF_Section_Info', 'render'],
            ],
        ];

        // Render Sidebar Layout (Inline)
        ?>
        <div class="cmbsqf-settings-layout">
            
            <!-- Sidebar Navigation -->
            <nav class="cmbsqf-sidebar-nav">
                <ul class="cmbsqf-sidebar-menu">
                    <?php 
                    $first = true;
                    foreach ($sections as $id => $section): 
                        $active_class = $first ? ' active' : '';
                        $first = false;
                    ?>
                        <li class="cmbsqf-sidebar-item<?php echo esc_attr($active_class); ?>">
                            <a href="#" class="cmbsqf-sidebar-link" data-section="<?php echo esc_attr($id); ?>">
                                <?php if (!empty($section['icon'])): ?>
                                    <span class="cmbsqf-sidebar-icon"><?php echo wp_kses_post($section['icon']); ?></span>
                                <?php endif; ?>
                                <span class="cmbsqf-sidebar-label"><?php echo esc_html($section['label']); ?></span>
                            </a>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </nav>

            <!-- Section Content -->
            <section class="cmbsqf-settings-content">
                <?php 
                $first = true;
                foreach ($sections as $id => $section): 
                    $display_style = $first ? 'block' : 'none';
                    $first = false;
                ?>
                    <div class="cmbsqf-section-content" data-section-id="<?php echo esc_attr($id); ?>" style="display: <?php echo esc_attr($display_style); ?>;">
                        <?php 
                        if (isset($section['callback']) && is_callable($section['callback'])) {
                            call_user_func($section['callback'], $form_id);
                        }
                        ?>
                    </div>
                <?php endforeach; ?>
            </section>

        </div>
        <?php
    }

    /**
     * Get SVG icon markup for sections.
     *
     * @param string $icon_name Icon identifier.
     * @return string SVG markup.
     */
    private static function get_icon($icon_name) {
        $icons = [
            'behavior' => '<svg viewBox="0 0 24 24"><path d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4"></path></svg>',
            'data'     => '<svg viewBox="0 0 24 24"><ellipse cx="12" cy="5" rx="9" ry="3"></ellipse><path d="M21 12c0 1.66-4 3-9 3s-9-1.34-9-3"></path><path d="M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5"></path></svg>',
            'info'     => '<svg viewBox="0 0 24 24"><circle cx="12" cy="12" r="10"></circle><path d="M12 16v-4m0-4h.01"></path></svg>',
        ];

        return $icons[$icon_name] ?? '';
    }
}
