<?php
/**
 * Styling Section Renderer
 *
 * Purpose: Renders the Styling section (Form tab)
 * Location: /admin/sections/class-section-styling.php
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once CMBSQF_PLUGIN_DIR . 'admin/helpers/class-helper-override-box.php';
require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';

/**
 * Class CMBSQF_Section_Styling
 *
 * Renders styling and CSS customization section.
 */
class CMBSQF_Section_Styling {

    /**
     * Render the Styling section.
     *
     * @param int $form_id Form ID (0 for global, specific ID for form).
     */
    public static function render($form_id = 0) {
        // Check if CSS editor is enabled
        if (defined('CMBSQF_ENABLE_CSS_EDITOR') && CMBSQF_ENABLE_CSS_EDITOR) {
            // Full CSS editor (personal use)
            self::render_with_css_editor($form_id);
        } else {
            // Simple template selector (WordPress.org compliance)
            self::render_without_css_editor($form_id);
        }
    }

    /**
     * Render without CSS editor - simple template selector (WordPress.org version).
     *
     * This version complies with WordPress.org guidelines that prohibit
     * arbitrary CSS/JS/PHP insertion by users.
     *
     * @param int $form_id Form ID.
     */
    private static function render_without_css_editor($form_id = 0) {
        $context = $form_id > 0 ? 'form' : 'global';
        $templates = self::get_css_templates();
        
        // Get current template
        $current_template = CMBSQF_DB_Settings::get('form__styling__template', $form_id, 'default');
        
        ?>
        <form class="cmbsqf-settings-form" data-context="<?php echo esc_attr($context); ?>" <?php echo $form_id > 0 ? 'data-form-id="' . esc_attr($form_id) . '"' : ''; ?>>

            <?php
            // Render Override Box (only for form context)
            if ($form_id > 0) {
                $override = CMBSQF_DB_Settings::get('form__styling__override', $form_id, '0');
                CMBSQF_Helper_Override_Box::render('styling', $override, $form_id);
            }
            ?>

            <div <?php echo $form_id > 0 ? 'data-toggle-target="styling-fields"' : ''; ?>>
                
                <h4><?php esc_html_e('Select Form Style', 'cmb-sqlite-form'); ?></h4>
                
                <p class="description">
                    <?php esc_html_e('Choose a pre-designed template for your form styling.', 'cmb-sqlite-form'); ?>
                </p>

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="form__styling__template">
                                <?php esc_html_e('Style Template', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <select id="form__styling__template" name="form__styling__template" class="regular-text">
                                <?php foreach ($templates as $key => $template): ?>
                                    <option value="<?php echo esc_attr($key); ?>" <?php selected($current_template, $key); ?>>
                                        <?php echo esc_html($template['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description">
                                <?php
                                printf(
                                    /* translators: %s: HTML link to WordPress Customizer */
                                    esc_html__('For advanced customization, use %s.', 'cmb-sqlite-form'),
                                    '<strong>' . esc_html__('Appearance > Customize > Additional CSS', 'cmb-sqlite-form') . '</strong>'
                                );
                                ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <details style="margin-top: 20px;">
                    <summary style="cursor: pointer; user-select: none; font-weight: 500;">
                        <?php esc_html_e('Available CSS Classes', 'cmb-sqlite-form'); ?>
                    </summary>
                    <div style="margin-top: 10px; padding: 15px; background: #f9f9f9; border-left: 3px solid #2271b1;">
                        <p><?php esc_html_e('Use these classes for custom CSS in the WordPress Customizer:', 'cmb-sqlite-form'); ?></p>
                        <ul style="list-style: disc; margin-left: 20px;">
                            <li><code>.cmbsqf-wrap</code> - <?php esc_html_e('Form container', 'cmb-sqlite-form'); ?></li>
                            <li><code>.cmbsqf-title</code> - <?php esc_html_e('Form title', 'cmb-sqlite-form'); ?></li>
                            <li><code>.cmbsqf-field</code> - <?php esc_html_e('Field wrapper', 'cmb-sqlite-form'); ?></li>
                            <li><code>.cmbsqf-label</code> - <?php esc_html_e('Field label', 'cmb-sqlite-form'); ?></li>
                            <li><code>.cmbsqf-input</code> - <?php esc_html_e('Text inputs', 'cmb-sqlite-form'); ?></li>
                            <li><code>.cmbsqf-textarea</code> - <?php esc_html_e('Textarea fields', 'cmb-sqlite-form'); ?></li>
                            <li><code>.cmbsqf-button</code> - <?php esc_html_e('Submit button', 'cmb-sqlite-form'); ?></li>
                            <li><code>.cmbsqf-message</code> - <?php esc_html_e('Message container', 'cmb-sqlite-form'); ?></li>
                        </ul>
                    </div>
                </details>

            </div>

            <p class="submit">
                <button type="submit" class="button button-primary">
                    <?php esc_html_e('Save Styling Settings', 'cmb-sqlite-form'); ?>
                </button>
            </p>
        </form>
        <?php
    }

    /**
     * Render with CSS editor - full custom CSS editor (personal use version).
     *
     * This version includes a full CSS editor for custom styling.
     * Disabled by default via CMBSQF_ENABLE_CSS_EDITOR constant.
     *
     * @param int $form_id Form ID.
     */
    private static function render_with_css_editor($form_id = 0) {
        $context = $form_id > 0 ? 'form' : 'global';

        // Load dependencies
        require_once CMBSQF_PLUGIN_DIR . 'includes/class-css-storage.php';

        // Load values
        // Custom CSS is stored in a file for both global and form-specific contexts
        $custom_css = CMBSQF_CSS_Storage::get($form_id);
        
        // Fallback to DB if file is empty or unavailable
        if (empty($custom_css)) {
             $custom_css = CMBSQF_DB_Settings::get('form__styling__custom_css', $form_id, '');
        }

        // Get available CSS templates
        $templates = self::get_css_templates();

        ?>
        <form class="cmbsqf-settings-form" data-context="<?php echo esc_attr($context); ?>" <?php echo $form_id > 0 ? 'data-form-id="' . esc_attr($form_id) . '"' : ''; ?>>

            <?php
            // Render Override Box (only for form context)
            if ($form_id > 0) {
                $override = CMBSQF_DB_Settings::get('form__styling__override', $form_id, '0');
                CMBSQF_Helper_Override_Box::render('styling', $override, $form_id);
            }
            ?>

            <div <?php echo $form_id > 0 ? 'data-toggle-target="styling-fields"' : ''; ?>>
                
                <!-- Custom CSS Editor -->
                <h4><?php esc_html_e('Custom CSS', 'cmb-sqlite-form'); ?></h4>
                <textarea id="form__styling__custom_css"
                          name="form__styling__custom_css"
                          rows="20"
                          class="large-text code cmbsqf-css-editor"
                          style="width: 100%; max-width: 100%;"
                          spellcheck="false"><?php echo esc_textarea($custom_css); ?></textarea>
                <p class="description">
                    <?php esc_html_e('Global default CSS that will be used by all forms unless overridden per-form.', 'cmb-sqlite-form'); ?>
                    <br>
                    <strong><?php esc_html_e('Available CSS classes:', 'cmb-sqlite-form'); ?></strong>
                    <code>.cmbsqf-wrap</code>, <code>.cmbsqf-title</code>, <code>.cmbsqf-field</code>, <code>.cmbsqf-label</code>,
                    <code>.cmbsqf-input</code>, <code>.cmbsqf-textarea</code>, <code>.cmbsqf-button</code>, <code>.cmbsqf-message</code>,
                    <code>.cmbsqf-success</code>, <code>.cmbsqf-error</code>
                </p>

                <!-- Template Selector (Collapsible) -->
                <details style="margin-top: 20px;">
                    <summary style="cursor: pointer; user-select: none; font-weight: 500;">
                        <?php esc_html_e('Start from a template', 'cmb-sqlite-form'); ?>
                    </summary>
                    <div style="margin-top: 10px; padding: 15px; background: #f9f9f9; border-left: 3px solid #2271b1;">
                        <select id="cmb_sf_template_select" class="regular-text cmbsqf-template-select">
                            <option value=""><?php esc_html_e('-- Select a template --', 'cmb-sqlite-form'); ?></option>
                            <?php foreach ($templates as $key => $template): ?>
                                <option value="<?php echo esc_attr($key); ?>" data-css="<?php echo esc_attr($template['css']); ?>">
                                    <?php echo esc_html($template['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <button type="button" id="cmb_sf_load_template_btn" class="button cmbsqf-template-btn" style="margin-left: 5px;">
                            <?php esc_html_e('Load Selected Template', 'cmb-sqlite-form'); ?>
                        </button>
                        <p class="description" style="margin-top: 8px;">
                            <?php esc_html_e('Select a predefined template and click "Load" to apply it to the CSS editor above.', 'cmb-sqlite-form'); ?>
                        </p>
                    </div>
                </details>

            </div>

            <p class="submit">
                <button type="submit" class="button button-primary">
                    <?php esc_html_e('Save Styling Settings', 'cmb-sqlite-form'); ?>
                </button>
            </p>
        </form>
        <?php
    }

    /**
     * Get available CSS templates.
     *
     * @return array Template definitions with CSS content.
     */
    private static function get_css_templates() {
        $templates_dir = CMBSQF_PLUGIN_DIR . 'admin/css-templates/';
        $templates = [];
        
        // Scan directory for .css files
        $files = glob($templates_dir . '*.css');
        
        if ($files) {
            foreach ($files as $file) {
                $key = basename($file, '.css');
                // Generate human-readable name from filename
                $name = ucwords(str_replace(['-', '_'], ' ', $key));
                
                $templates[$key] = [
                    'name' => $name,
                    'css'  => file_get_contents($file)
                ];
            }
        }
        
        return $templates;
    }
}
