<?php
/**
 * Outgoing Server Section Renderer
 *
 * Purpose: Renders the Outgoing Server section (Notifications tab)
 * Location: /admin/sections/class-section-outgoing-server.php
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once CMBSQF_PLUGIN_DIR . 'admin/helpers/class-helper-override-box.php';
require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';

/**
 * Class CMBSQF_Section_Outgoing_Server
 *
 * Renders SMTP server configuration section.
 */
class CMBSQF_Section_Outgoing_Server {

    /**
     * Render the Outgoing Server section.
     *
     * @param int $form_id Form ID (0 for global, specific ID for form).
     */
    public static function render($form_id = 0) {
        $context = $form_id > 0 ? 'form' : 'global';

        // Load values from database
        $smtp_enabled    = CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_enabled', $form_id, '0');
        $smtp_host       = CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_host', $form_id, '');
        $smtp_port       = CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_port', $form_id, '587');
        $smtp_username   = CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_username', $form_id, '');
        $smtp_password   = CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_password', $form_id, '');
        $smtp_encryption = CMBSQF_DB_Settings::get('notifications__outgoing_server__smtp_encryption', $form_id, 'tls');

        // Sender Identity
        $from_name  = CMBSQF_DB_Settings::get('notifications__sender__from_name', $form_id, get_bloginfo('name'));
        $from_email = CMBSQF_DB_Settings::get('notifications__sender__from_email', $form_id, get_bloginfo('admin_email'));

        ?>
        <form class="cmbsqf-settings-form" data-context="<?php echo esc_attr($context); ?>" <?php echo $form_id > 0 ? 'data-form-id="' . esc_attr($form_id) . '"' : ''; ?>>

            <?php
            // Render Override Box (only for form context)
            if ($form_id > 0) {
                $override = CMBSQF_DB_Settings::get('notifications__outgoing_server__override', $form_id, '0');
                CMBSQF_Helper_Override_Box::render('notifications__outgoing_server', $override, $form_id);
            }
            ?>

            <div <?php echo $form_id > 0 ? 'data-toggle-target="notifications--outgoing-server-fields"' : ''; ?>>
            <table class="form-table" role="presentation">
                <tbody>
                    <!-- Sender Identity -->
                    <tr>
                        <th scope="row">
                            <label for="notifications__sender__from_name">
                                <?php esc_html_e('From Name', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="text"
                                   id="notifications__sender__from_name"
                                   name="notifications__sender__from_name"
                                   value="<?php echo esc_attr($from_name); ?>"
                                   class="regular-text" />
                            <p class="description">
                                <?php esc_html_e('Name that appears in the "From" field of outgoing emails.', 'cmb-sqlite-form'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="notifications__sender__from_email">
                                <?php esc_html_e('From Email', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="email"
                                   id="notifications__sender__from_email"
                                   name="notifications__sender__from_email"
                                   value="<?php echo esc_attr($from_email); ?>"
                                   class="regular-text" />
                            <p class="description">
                                <?php esc_html_e('Email address that appears in the "From" field of outgoing emails.', 'cmb-sqlite-form'); ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Mailing Method -->
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Email Delivery Method', 'cmb-sqlite-form'); ?>
                        </th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="radio"
                                           name="notifications__outgoing_server__smtp_enabled"
                                           value="0"
                                           <?php checked($smtp_enabled, '0'); ?> />
                                    <?php esc_html_e('Use WordPress default (wp_mail)', 'cmb-sqlite-form'); ?>
                                </label>
                                <br>
                                <label>
                                    <input type="radio"
                                           name="notifications__outgoing_server__smtp_enabled"
                                           value="1"
                                           data-toggle-group="smtp-fields"
                                           data-toggle-value="1"
                                           <?php checked($smtp_enabled, '1'); ?> />
                                    <?php esc_html_e('Use custom SMTP server', 'cmb-sqlite-form'); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e('Choose how emails should be sent. Using SMTP is recommended for reliable email delivery.', 'cmb-sqlite-form'); ?>
                                </p>
                            </fieldset>
                        </td>
                    </tr>
                    <tr data-toggle-target="smtp-fields">
                        <th scope="row">
                            <label for="notifications__outgoing_server__smtp_host">
                                <?php esc_html_e('SMTP Host', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="text"
                                   id="notifications__outgoing_server__smtp_host"
                                   name="notifications__outgoing_server__smtp_host"
                                   value="<?php echo esc_attr($smtp_host); ?>"
                                   class="regular-text"
                                   placeholder="smtp.example.com" />
                            <p class="description">
                                <?php esc_html_e('SMTP server hostname or IP address.', 'cmb-sqlite-form'); ?>
                                <br>
                                <?php esc_html_e('Example: smtp.gmail.com, smtp.sendgrid.net', 'cmb-sqlite-form'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr data-toggle-target="smtp-fields">
                        <th scope="row">
                            <label for="notifications__outgoing_server__smtp_port">
                                <?php esc_html_e('SMTP Port', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="number"
                                   id="notifications__outgoing_server__smtp_port"
                                   name="notifications__outgoing_server__smtp_port"
                                   value="<?php echo esc_attr($smtp_port); ?>"
                                   class="small-text"
                                   min="1"
                                   max="65535" />
                            <p class="description">
                                <?php esc_html_e('Common ports: 25 (default), 465 (SSL), 587 (TLS)', 'cmb-sqlite-form'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr data-toggle-target="smtp-fields">
                        <th scope="row">
                            <label for="notifications__outgoing_server__smtp_encryption">
                                <?php esc_html_e('Encryption', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <select id="notifications__outgoing_server__smtp_encryption"
                                    name="notifications__outgoing_server__smtp_encryption">
                                <option value="tls" <?php selected($smtp_encryption, 'tls'); ?>>TLS</option>
                                <option value="ssl" <?php selected($smtp_encryption, 'ssl'); ?>>SSL</option>
                                <option value="none" <?php selected($smtp_encryption, 'none'); ?>><?php esc_html_e('None', 'cmb-sqlite-form'); ?></option>
                            </select>
                            <p class="description">
                                <?php esc_html_e('Encryption method for secure connection. TLS is recommended.', 'cmb-sqlite-form'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr data-toggle-target="smtp-fields">
                        <th scope="row">
                            <label for="notifications__outgoing_server__smtp_username">
                                <?php esc_html_e('SMTP Username', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="text"
                                   id="notifications__outgoing_server__smtp_username"
                                   name="notifications__outgoing_server__smtp_username"
                                   value="<?php echo esc_attr($smtp_username); ?>"
                                   class="regular-text"
                                   autocomplete="off" />
                            <p class="description">
                                <?php esc_html_e('Username for SMTP authentication. Leave empty if not required.', 'cmb-sqlite-form'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr data-toggle-target="smtp-fields">
                        <th scope="row">
                            <label for="notifications__outgoing_server__smtp_password">
                                <?php esc_html_e('SMTP Password', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="password"
                                   id="notifications__outgoing_server__smtp_password"
                                   name="notifications__outgoing_server__smtp_password"
                                   value="<?php echo esc_attr($smtp_password); ?>"
                                   class="regular-text"
                                   autocomplete="new-password"
                                   placeholder="<?php echo empty($smtp_password) ? '' : '••••••••'; ?>" />
                            <p class="description">
                                <?php esc_html_e('Password for SMTP authentication. Leave empty to keep existing password.', 'cmb-sqlite-form'); ?>
                                <?php if (!empty($smtp_password)) : ?>
                                    <br>
                                    <strong><?php esc_html_e('Note: Password is currently set. Enter a new password to change it.', 'cmb-sqlite-form'); ?></strong>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>

                    <!-- Connection Test -->
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Connection Test', 'cmb-sqlite-form'); ?>
                        </th>
                        <td>
                            <div style="display: flex; gap: 10px; align-items: flex-start; margin-bottom: 10px;">
                                <input type="email" 
                                       id="cmbsqf_test_email_recipient" 
                                       value="<?php echo esc_attr(get_option('admin_email')); ?>" 
                                       class="regular-text" 
                                       placeholder="<?php esc_attr_e('recipient@example.com', 'cmb-sqlite-form'); ?>" />
                                
                                <button type="button" id="cmbsqf_send_test_email" class="button button-secondary">
                                    <?php esc_html_e('Send Test Email', 'cmb-sqlite-form'); ?>
                                </button>
                                
                                <span class="spinner" style="float: none; margin: 0;"></span>
                            </div>
                            
                            <p class="description">
                                <?php esc_html_e('Send a test email using the current settings above (without saving). Useful to verify SMTP configuration.', 'cmb-sqlite-form'); ?>
                            </p>
                            
                            <div id="cmbsqf_test_email_result" style="margin-top: 10px; display: none;"></div>
                        </td>
                    </tr>
                </tbody>
            </table>
            </div>

            <p class="submit">
                <button type="submit" class="button button-primary">
                    <?php esc_html_e('Save Outgoing Server Settings', 'cmb-sqlite-form'); ?>
                </button>
            </p>
        </form>
        <?php
    }
}
