<?php
/**
 * Junk Email Protection Section Renderer
 *
 * Purpose: Renders the Junk Email Protection section (General tab)
 * Location: /admin/sections/class-section-junk-email.php
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load dependencies
require_once CMBSQF_PLUGIN_DIR . 'admin/helpers/class-helper-override-box.php';
require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';

/**
 * Class CMBSQF_Section_Junk_Email
 *
 * Renders junk email protection section.
 */
class CMBSQF_Section_Junk_Email {

    /**
     * Render the Junk Email Protection section.
     *
     * @param int $form_id Form ID (always 0 for global-only section).
     */
    public static function render($form_id = 0) {
        $context = $form_id > 0 ? 'form' : 'global';

        // Load values from database
        $block_disposable   = CMBSQF_DB_Settings::get('form__junk_email__block_disposable', $form_id, '0');
        $disposable_message = CMBSQF_DB_Settings::get('form__junk_email__disposable_message', $form_id, 'Disposable or temporary email addresses are not allowed. Please use a permanent email address.');

        ?>
        <form class="cmbsqf-settings-form" data-context="<?php echo esc_attr($context); ?>" <?php echo $form_id > 0 ? 'data-form-id="' . esc_attr($form_id) . '"' : ''; ?>>

            <?php
            // Render Override Box (only for form context)
            if ($form_id > 0) {
                $override = CMBSQF_DB_Settings::get('form__junk_email__override', $form_id, '0');
                CMBSQF_Helper_Override_Box::render('junk_email', $override, $form_id);
            }
            ?>

            <div <?php echo $form_id > 0 ? 'data-toggle-target="junk-email-fields"' : ''; ?>>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row">
                            <?php esc_html_e('Block Disposable Emails', 'cmb-sqlite-form'); ?>
                        </th>
                        <td>
                            <fieldset>
                                <label for="form__junk_email__block_disposable">
                                    <input type="checkbox"
                                           id="form__junk_email__block_disposable"
                                           name="form__junk_email__block_disposable"
                                           value="1"
                                           data-toggle-group="disposable-email-fields"
                                           <?php checked($block_disposable, '1'); ?> />
                                    <?php esc_html_e('Block submissions from disposable/temporary email addresses', 'cmb-sqlite-form'); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e('When enabled, submissions using disposable email services (like Mailinator, Guerrilla Mail, etc.) will be rejected.', 'cmb-sqlite-form'); ?>
                                </p>
                            </fieldset>
                        </td>
                    </tr>
                    <tr data-toggle-target="disposable-email-fields">
                        <th scope="row">
                            <label for="form__junk_email__disposable_message">
                                <?php esc_html_e('Error Message', 'cmb-sqlite-form'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="text"
                                   id="form__junk_email__disposable_message"
                                   name="form__junk_email__disposable_message"
                                   value="<?php echo esc_attr($disposable_message); ?>"
                                   class="large-text" />
                            <p class="description">
                                <?php esc_html_e('Message shown to users when a disposable email address is detected.', 'cmb-sqlite-form'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr data-toggle-target="disposable-email-fields">
                        <th scope="row">
                            <?php esc_html_e('Blocklist Sources', 'cmb-sqlite-form'); ?>
                        </th>
                        <td>
                            <p class="description">
                                <?php esc_html_e('URLs to domain blocklists. The plugin will download and combine these sources into a single deduplicated list.', 'cmb-sqlite-form'); ?>
                            </p>

                            <?php
                            // Load sources and metadata from database
                            $sources = CMBSQF_DB_Settings::get('form__junk_email__sources', $form_id, '[]');
                            $sources = json_decode($sources, true);
                            if (!is_array($sources)) {
                                $sources = [];
                            }

                            $metadata = CMBSQF_DB_Settings::get('form__junk_email__sources_metadata', $form_id, '[]');
                            // Pass metadata to JS via data attribute
                            
                            $total_domains = CMBSQF_DB_Settings::get('form__junk_email__total_domains', $form_id, '0');
                            $last_update = CMBSQF_DB_Settings::get('form__junk_email__last_update', $form_id, '0');
                            ?>

                            <!-- Add New Source -->
                            <div class="cmbsqf-add-source-wrapper" style="display: flex; gap: 10px; margin-bottom: 15px;">
                                <input type="url" 
                                       id="cmbsqf-new-source-url" 
                                       class="large-text code" 
                                       placeholder="https://example.com/blocklist.txt" 
                                       style="flex: 1;">
                                <button type="button" class="button button-secondary" id="cmbsqf-add-source-btn">
                                    <?php esc_html_e('Add Source', 'cmb-sqlite-form'); ?>
                                </button>
                            </div>

                            <!-- Active Sources List -->
                            <div class="cmbsqf-sources-list" id="cmbsqf-sources-list" data-metadata="<?php echo esc_attr($metadata); ?>">
                                <?php foreach ($sources as $source_url) : ?>
                                    <div class="cmbsqf-source-item" data-url="<?php echo esc_attr($source_url); ?>">
                                        <code class="cmbsqf-source-url"><?php echo esc_html($source_url); ?></code>
                                        <div class="cmbsqf-source-actions">
                                            <button type="button" class="button button-small cmbsqf-source-info" title="<?php esc_attr_e('View Info', 'cmb-sqlite-form'); ?>">
                                                <span class="dashicons dashicons-info"></span>
                                            </button>
                                            <button type="button" class="button button-small cmbsqf-remove-source" title="<?php esc_attr_e('Remove', 'cmb-sqlite-form'); ?>">
                                                <span class="dashicons dashicons-trash"></span>
                                            </button>
                                        </div>
                                        <input type="hidden" name="form__junk_email__sources[]" value="<?php echo esc_attr($source_url); ?>">
                                    </div>
                                <?php endforeach; ?>
                            </div>

                            <!-- Info Panel (Hidden by default) -->
                            <div id="cmbsqf-source-info-panel" class="notice notice-info inline" style="display: none; margin: 10px 0;">
                                <p>
                                    <strong><?php esc_html_e('Source Details:', 'cmb-sqlite-form'); ?></strong><br>
                                    <span class="cmbsqf-info-content"></span>
                                </p>
                            </div>

                            <details class="cmbsqf-examples-details" style="margin-top: 15px;">
                                <summary class="cmbsqf-examples-summary"><?php esc_html_e('Show popular sources', 'cmb-sqlite-form'); ?></summary>
                                <div class="cmbsqf-examples-content">
                                    <p class="description">
                                        <strong><?php esc_html_e('Community Standard', 'cmb-sqlite-form'); ?></strong> &ndash; 
                                        <?php esc_html_e('Widely used, community-validated list.', 'cmb-sqlite-form'); ?>
                                    </p>
                                    <code class="cmbsqf-code-example">https://raw.githubusercontent.com/disposable-email-domains/disposable-email-domains/master/disposable_email_blocklist.conf</code>

                                    <p class="description" style="margin-top: 15px;">
                                        <strong><?php esc_html_e('Daily Updates', 'cmb-sqlite-form'); ?></strong> &ndash; 
                                        <?php esc_html_e('Comprehensive list updated automatically every day.', 'cmb-sqlite-form'); ?>
                                    </p>
                                    <code class="cmbsqf-code-example">https://raw.githubusercontent.com/disposable/disposable-email-domains/master/domains.txt</code>

                                    <p class="description" style="margin-top: 15px;">
                                        <strong><?php esc_html_e('MX Validated', 'cmb-sqlite-form'); ?></strong> &ndash; 
                                        <?php esc_html_e('Checked against DNS records to ensure domains are active.', 'cmb-sqlite-form'); ?>
                                    </p>
                                    <code class="cmbsqf-code-example">https://raw.githubusercontent.com/groundcat/disposable-email-domain-list/master/domains.txt</code>
                                </div>
                            </details>

                            <div class="cmbsqf-blocklist-combined" style="margin-top: 20px; padding: 15px; background: #f0f0f1; border-left: 4px solid #2271b1;">
                                <strong><?php esc_html_e('Combined Blocklist', 'cmb-sqlite-form'); ?></strong>
                                <p style="margin: 10px 0 0 0;">
                                    <?php
                                    if ($total_domains > 0) {
                                        printf(
                                            /* translators: %s: formatted number of unique domains in the blocklist */
                                            esc_html__('Total unique domains: %s', 'cmb-sqlite-form'),
                                            '<strong>' . number_format($total_domains) . '</strong>'
                                        );
                                        echo '<br>';
                                        if ($last_update > 0) {
                                            printf(
                                                /* translators: %s: time elapsed since last blocklist update (e.g., "2 hours ago") */
                                                esc_html__('Last updated: %s', 'cmb-sqlite-form'),
                                                '<strong>' . esc_html(human_time_diff($last_update, time()) . ' ' . __('ago', 'cmb-sqlite-form')) . '</strong>'
                                            );
                                        }
                                    } else {
                                        esc_html_e('No blocklist loaded yet. Click "Update Blocklist Now" to download sources.', 'cmb-sqlite-form');
                                    }
                                    ?>
                                </p>
                                <button type="button" class="button button-primary cmbsqf-update-blocklist" style="margin-top: 10px;" data-form-id="<?php echo esc_attr($form_id); ?>">
                                    <span class="dashicons dashicons-update" style="margin-top: 3px;"></span>
                                    <?php esc_html_e('Update Blocklist Now', 'cmb-sqlite-form'); ?>
                                </button>
                                <div class="cmbsqf-blocklist-status" style="margin-top: 10px;"></div>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
            </div>

            <p class="submit">
                <button type="submit" class="button button-primary">
                    <?php esc_html_e('Save Junk Email Settings', 'cmb-sqlite-form'); ?>
                </button>
            </p>
        </form>
        <?php
    }
}
