<?php
/**
 * Admin list table for All Forms.
 *
 * Extends WP_List_Table to provide a WordPress-standard interface
 * for managing forms with search, pagination, and bulk actions.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load WP_List_Table if not already loaded
if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Class CMBSQF_Page_Forms
 *
 * Custom WP_List_Table implementation for displaying forms.
 * Includes rendering methods for the full forms list page.
 */
class CMBSQF_Page_Forms extends WP_List_Table {

        /**
         * Total number of items.
         *
         * @var int
         */
        public $total_items = 0;

        /**
         * Constructor.
         */
        public function __construct() {
            parent::__construct([
                'singular' => 'form',
                'plural'   => 'forms',
                'ajax'     => false
            ]);
        }

        /**
         * Generate the table row.
         *
         * @param object|array $item The current item.
         */
        public function single_row($item) {
            echo '<tr id="form-row-' . intval($item['id']) . '">';
            $this->single_row_columns($item);
            echo '</tr>';
        }

        /**
         * Define table columns.
         *
         * @return array Column definitions.
         */
        public function get_columns() {
            return [
                'cb'        => '<input type="checkbox" />',
                'id'        => __('ID', 'cmb-sqlite-form'),
                'form_name' => __('Form Name', 'cmb-sqlite-form'),
                'entries'   => __('Entries', 'cmb-sqlite-form'),
                'shortcode' => __('Shortcode', 'cmb-sqlite-form'),
            ];
        }

        /**
         * Define sortable columns.
         *
         * @return array Sortable column definitions.
         */
        public function get_sortable_columns() {
            return [
                'id'        => ['id', false],
                'form_name' => ['name', false],
                'entries'   => ['entries', false],
            ];
        }

        /**
         * Define bulk actions.
         *
         * @return array Bulk action definitions.
         */
        public function get_bulk_actions() {
            return [
                'delete' => __('Delete', 'cmb-sqlite-form'),
            ];
        }

        /**
         * Render the checkbox column.
         *
         * @param array $item Current item.
         * @return string Checkbox HTML.
         */
        public function column_cb($item) {
            return sprintf(
                '<input type="checkbox" name="form_ids[]" value="%d" />',
                intval($item['id'])
            );
        }

        /**
         * Render the ID column.
         *
         * @param array $item Current item.
         * @return string Column HTML.
         */
        public function column_id($item) {
            return intval($item['id']);
        }

        /**
         * Render the form name column with row actions.
         *
         * @param array $item Current item.
         * @return string Column HTML with row actions.
         */
        public function column_form_name($item) {
            $form_id = intval($item['id']);
            $form_name = esc_html($item['name']);

            // Build row actions
            $actions = [];

            // View Entries
            $actions['view_entries'] = sprintf(
                '<a href="%s">%s</a>',
                esc_url(add_query_arg(
                    [
                        'page'    => CMBSQF_Constants::SLUG_FORMS,
                        'view'    => 'entries',
                        'form_id' => $form_id,
                    ],
                    admin_url('admin.php')
                )),
                __('View Entries', 'cmb-sqlite-form')
            );

            // Settings
            $actions['settings'] = sprintf(
                '<a href="%s">%s</a>',
                esc_url(add_query_arg(
                    [
                        'page'    => CMBSQF_Constants::SLUG_FORMS,
                        'view'    => 'settings',
                        'form_id' => $form_id,
                    ],
                    admin_url('admin.php')
                )),
                __('Settings', 'cmb-sqlite-form')
            );

            // Rename
            $actions['rename'] = sprintf(
                '<a href="#" class="cmbsqf-rename-form" data-form-id="%d" data-form-name="%s">%s</a>',
                $form_id,
                esc_attr($item['name']),
                __('Rename', 'cmb-sqlite-form')
            );

            // Delete
            $actions['delete'] = sprintf(
                '<a href="%s" class="cmbsqf-delete-form submitdelete" data-form-id="%d" data-form-name="%s">%s</a>',
                esc_url(wp_nonce_url(
                    admin_url('admin.php?page=' . CMBSQF_Constants::SLUG_FORMS . '&action=delete&form_id=' . $form_id),
                    'delete_form_' . $form_id
                )),
                $form_id,
                esc_attr($item['name']),
                __('Delete', 'cmb-sqlite-form')
            );

            return sprintf(
                '<strong><a href="%s">%s</a></strong>%s',
                esc_url(add_query_arg(
                    [
                        'page'    => CMBSQF_Constants::SLUG_FORMS,
                        'view'    => 'entries',
                        'form_id' => $form_id,
                    ],
                    admin_url('admin.php')
                )),
                $form_name,
                $this->row_actions($actions)
            );
        }

        /**
     * Render the shortcode column.
     *
     * @param array $item Current item.
     * @return string Shortcode HTML with copy icon.
     */
    public function column_shortcode($item) {
        $shortcode = esc_html($item['shortcode']);
        
        $copy_icon = '<svg class="cmbsqf-icon-copy" width="16" height="16" viewBox="0 0 16 16" fill="none"><rect x="5.5" y="5.5" width="9" height="9" rx="1.5" stroke="currentColor" stroke-width="1.5"/><path d="M10.5 5.5V3a1.5 1.5 0 0 0-1.5-1.5H3A1.5 1.5 0 0 0 1.5 3v6A1.5 1.5 0 0 0 3 10.5h2.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"/></svg>';
        
        $check_icon = '<svg class="cmbsqf-icon-check" width="16" height="16" viewBox="0 0 16 16" fill="none" style="display:none;"><path d="M3 8l3 3 7-7" stroke="#46b450" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>';
        
        return sprintf(
            '<code style="user-select:all;background:#f0f0f1;padding:3px 6px;border-radius:3px;">%s</code> <span class="cmbsqf-copy-shortcode" data-shortcode="%s" title="%s" style="cursor:pointer;display:inline-block;vertical-align:middle;line-height:1;">%s%s</span>',
            $shortcode,
            esc_attr($item['shortcode']),
            esc_attr__('Copy', 'cmb-sqlite-form'),
            wp_kses_post($copy_icon),
            wp_kses_post($check_icon)
        );
    }

        /**
         * Render the entries column.
         *
         * @param array $item Current item.
         * @return string Formatted entry count.
         */
        public function column_entries($item) {
            return number_format_i18n(intval($item['num_entries']));
        }

        /**
         * Default column renderer.
         *
         * @param array  $item        Current item.
         * @param string $column_name Column name.
         * @return string Column content.
         */
        public function column_default($item, $column_name) {
            return esc_html($item[$column_name] ?? '');
        }

        /**
         * Prepare items for display.
         *
         * Handles pagination, sorting, and filtering.
         */
        public function prepare_items() {
            $per_page = 20;
            $current_page = $this->get_pagenum();

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Standard WP_List_Table query parameters for sorting/filtering
            $search = isset($_REQUEST['s']) ? sanitize_text_field(wp_unslash($_REQUEST['s'])) : '';

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Standard WP_List_Table query parameters for sorting/filtering
            $orderby = isset($_REQUEST['orderby']) ? sanitize_key(wp_unslash($_REQUEST['orderby'])) : 'name';

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Standard WP_List_Table query parameters for sorting/filtering
            $order = isset($_REQUEST['order']) && strtolower(sanitize_text_field(wp_unslash($_REQUEST['order']))) === 'desc' ? 'DESC' : 'ASC';

            $allowed = ['name', 'entries'];
            if (!in_array($orderby, $allowed, true)) {
                $orderby = 'name';
            }

            // Get all forms
            $all_forms = self::get_all_forms();

            // Apply search filter
            if ($search !== '') {
                $all_forms = array_filter($all_forms, function($form) use ($search) {
                    return stripos($form['name'], $search) !== false ||
                           stripos($form['shortcode'], $search) !== false ||
                           stripos((string)$form['id'], $search) !== false;
                });
            }

            // Apply sorting
            usort($all_forms, function($a, $b) use ($orderby, $order) {
                $val_a = $orderby === 'entries' ? $a['num_entries'] : $a['name'];
                $val_b = $orderby === 'entries' ? $b['num_entries'] : $b['name'];

                if ($orderby === 'entries') {
                    $result = $val_a - $val_b;
                } else {
                    $result = strcasecmp($val_a, $val_b);
                }

                return $order === 'DESC' ? -$result : $result;
            });

            $this->total_items = count($all_forms);

            // Apply pagination
            $offset = ($current_page - 1) * $per_page;
            $this->items = array_slice($all_forms, $offset, $per_page);

            $this->_column_headers = [$this->get_columns(), [], $this->get_sortable_columns()];
            $this->set_pagination_args([
                'total_items' => $this->total_items,
                'per_page'    => $per_page,
                'total_pages' => ceil(max(1, $this->total_items) / $per_page),
            ]);
        }

        /**
         * Get all forms from the database.
         *
         * @return array Array of form data.
         */
        private static function get_all_forms() {
            // Load DB Core
            if (!class_exists('CMBSQF_DB_Core')) {
                require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
            }

            return CMBSQF_DB_Core::get_all_forms();
        }

        /**
         * Message to display when no forms are found.
         */
        public function no_items() {
            esc_html_e('No forms found.', 'cmb-sqlite-form');
        }

        /**
         * Main entry point to render the forms list page.
         *
         * This static method orchestrates the entire page rendering process.
         */
        public static function render() {
            // Create and prepare the table
            $table = new self();
            $table->prepare_items();

            ?>
            <div class="wrap">
                <h1 class="wp-heading-inline"><?php esc_html_e('All Forms', 'cmb-sqlite-form'); ?></h1>
                <button type="button" class="page-title-action cmbsqf-create-form">
                    <?php esc_html_e('Add New', 'cmb-sqlite-form'); ?>
                </button>
                <hr class="wp-header-end">

                <form method="post">
                    <input type="hidden" name="page" value="<?php echo esc_attr(CMBSQF_Constants::SLUG_FORMS); ?>">
                    <?php
                    $table->search_box(__('Search Forms', 'cmb-sqlite-form'), 'cmbsqf-forms');
                    $table->display();
                    ?>
                </form>
            </div>
            <?php
        }

        /**
         * Delete a form and its associated data.
         *
         * @param int $form_id The form ID to delete.
         * @return bool True on success, false on failure.
         */
        private static function delete_form($form_id) {
            // Load DB Core
            if (!class_exists('CMBSQF_DB_Core')) {
                require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
            }

            return CMBSQF_DB_Core::delete_form($form_id);
        }

}
