<?php
/**
 * Admin page for viewing form entries
 *
 * Displays all submissions for a specific form using WP_List_Table
 */

if (!defined('ABSPATH')) {
    exit;
}

// Load WP_List_Table if not already loaded
if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class CMBSQF_Page_Form_Entries extends WP_List_Table {
    
    /**
     * Form ID for this entries view
     *
     * @var int
     */
    private $form_id;
    
    /**
     * Total number of items
     *
     * @var int
     */
    public $total_items = 0;
    
    /**
     * Constructor
     *
     * @param int $form_id Form ID
     */
    public function __construct($form_id) {
        parent::__construct([
            'singular' => 'entry',
            'plural'   => 'entries',
            'ajax'     => false
        ]);
        
        $this->form_id = $form_id;
    }
    
    /**
     * Define table columns
     *
     * @return array Column definitions
     */
    public function get_columns() {
        return [
            'cb'       => '<input type="checkbox" />',
            'id'       => __('ID', 'cmb-sqlite-form'),
            'name'     => __('Name', 'cmb-sqlite-form'),
            'email'    => __('Email', 'cmb-sqlite-form'),
            'message'  => __('Message', 'cmb-sqlite-form'),
            'date'     => __('Date', 'cmb-sqlite-form'),
            'notified' => __('Notified', 'cmb-sqlite-form'),
        ];
    }
    
    /**
     * Define sortable columns
     *
     * @return array Sortable column definitions
     */
    public function get_sortable_columns() {
        return [
            'id'    => ['id', false],
            'name'  => ['name', false],
            'email' => ['email', false],
            'date'  => ['created_at', true], // Default sort by date DESC
        ];
    }
    
    /**
     * Define bulk actions
     *
     * @return array Bulk action definitions
     */
    public function get_bulk_actions() {
        return [
            'resend_user_email' => __('Resend User Email', 'cmb-sqlite-form'),
            'delete'            => __('Delete', 'cmb-sqlite-form'),
        ];
    }
    
    /**
     * Render the checkbox column
     *
     * @param array $item Current item
     * @return string Checkbox HTML
     */
    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="entry_ids[]" value="%d" />',
            intval($item['id'])
        );
    }
    
    /**
     * Override single_row to add data attribute for delete animation
     *
     * @param array $item Current item
     */
    public function single_row($item) {
        echo sprintf('<tr data-entry-id="%d">', intval($item['id']));
        $this->single_row_columns($item);
        echo '</tr>';
    }
    
    /**
     * Render the ID column
     *
     * @param array $item Current item
     * @return string Column HTML
     */
    public function column_id($item) {
        return intval($item['id']);
    }
    
    /**
     * Render the name column
     *
     * @param array $item Current item
     * @return string Column HTML
     */
    public function column_name($item) {
        $name = trim($item['name'] . ' ' . $item['surname']);
        $display_name = $name ?: '';
        
        // Create link to open modal
        $link = sprintf(
            '<a href="#" class="cmbsqf-entry-link" data-entry-id="%d"><strong>%s</strong></a>',
            intval($item['id']),
            esc_html($display_name)
        );
        
        return $link;
    }
    
    /**
     * Render the email column
     *
     * @param array $item Current item
     * @return string Column HTML
     */
    public function column_email($item) {
        return esc_html($item['email'] ?? '');
    }
    
    /**
     * Render the message column (truncated)
     *
     * @param array $item Current item
     * @return string Column HTML
     */
    public function column_message($item) {
        if (empty($item['message'])) {
            return '';
        }
        
        $message = wp_strip_all_tags($item['message']);
        
        if (mb_strlen($message) > 100) {
            $message = mb_substr($message, 0, 100) . '...';
        }
        
        return esc_html($message);
    }
    
    /**
     * Render the date column
     *
     * @param array $item Current item
     * @return string Column HTML
     */
    public function column_date($item) {
        if (empty($item['created_at'])) {
            return '';
        }
        
        $timestamp = strtotime($item['created_at']);
        
        // Format: 2025-11-26 19:43:23 (ISO 8601 with seconds, 24-hour)
        return esc_html(gmdate('Y-m-d H:i:s', $timestamp));
    }
    
    /**
     * Render the notified status column
     *
     * @param array $item Current item
     * @return string Column HTML with colored bullet
     */
    public function column_notified($item) {
        $status = $item['email_delivery'] ?? 'not_configured';
        
        switch ($status) {
            case 'sent':
                return '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-sent" title="' . esc_attr__('Email sent successfully', 'cmb-sqlite-form') . '">&bull;</span>';
            
            case 'failed':
                $error = !empty($item['email_delivery_error']) ? esc_attr($item['email_delivery_error']) : __('Unknown error', 'cmb-sqlite-form');
                return '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-failed" title="' . esc_attr__('Failed: ', 'cmb-sqlite-form') . $error . '">&bull;</span>';
            
            case 'not_configured':
            default:
                return '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-na" title="' . esc_attr__('Email notification not configured', 'cmb-sqlite-form') . '">&bull;</span>';
        }
    }
    
    /**
     * Default column renderer
     *
     * @param array  $item        Current item
     * @param string $column_name Column name
     * @return string Column content
     */
    public function column_default($item, $column_name) {
        return esc_html($item[$column_name] ?? '');
    }
    
    /**
     * Prepare items for display
     *
     * Handles pagination, sorting, and filtering
     */
    public function prepare_items() {
        // Get per_page from request with validation
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- URL parameters for table display
        $per_page = isset($_REQUEST['per_page']) ? absint($_REQUEST['per_page']) : 25;
        $allowed_per_page = [10, 25, 50, 100];
        $per_page = in_array($per_page, $allowed_per_page, true) ? $per_page : 25;
        
        $current_page = $this->get_pagenum();
        
        // Get search query
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Search parameter for table filtering
        $search = isset($_REQUEST['s']) ? sanitize_text_field(wp_unslash($_REQUEST['s'])) : '';
        
        // Get sorting params
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Sorting parameters for table display
        $orderby = isset($_REQUEST['orderby']) ? sanitize_key(wp_unslash($_REQUEST['orderby'])) : 'created_at';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Sorting parameters for table display
        $order = isset($_REQUEST['order']) && strtolower(sanitize_text_field(wp_unslash($_REQUEST['order']))) === 'asc' ? 'ASC' : 'DESC';
        
        // Validate orderby
        $allowed = ['id', 'name', 'email', 'created_at'];
        if (!in_array($orderby, $allowed, true)) {
            $orderby = 'created_at';
        }
        
        // Get total count
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        $this->total_items = CMBSQF_DB_Submissions::get_count($this->form_id, $search);
        
        // Get items
        $offset = ($current_page - 1) * $per_page;
        $this->items = CMBSQF_DB_Submissions::get_by_form($this->form_id, $per_page, $offset, $orderby, $order, $search);
        
        // Set column headers
        $this->_column_headers = [$this->get_columns(), [], $this->get_sortable_columns()];
        
        // Set pagination
        $this->set_pagination_args([
            'total_items' => $this->total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil(max(1, $this->total_items) / $per_page),
        ]);
    }
    
    /**
     * Message to display when no entries are found
     */
    public function no_items() {
        esc_html_e('No entries found.', 'cmb-sqlite-form');
    }
    
    /**
     * Enqueue JavaScript and CSS for entries modal
     */
    private static function enqueue_assets($form_id) {
        // Enqueue entries controller (ES6 Module)
        wp_enqueue_script(
            'cmbsqf-entries-manager',
            CMBSQF_PLUGIN_URL . 'admin/assets/js/entries-controller.js',
            [],
            CMBSQF_VERSION,
            true
        );

        // Add global settings with nonces
        wp_add_inline_script(
            'cmbsqf-entries-manager',
            sprintf(
                'window.cmbsqfSettings = %s;',
                wp_json_encode([
                    'restUrl' => rest_url('cmbsqf/v1/admin'),
                    'nonce'   => wp_create_nonce('wp_rest'),
                    'formId'  => $form_id,
                    'downloadDbNonce'  => wp_create_nonce('cmbsqf_download_db'),
                    'downloadCsvNonce' => wp_create_nonce('cmbsqf_download_csv'),
                ])
            ),
            'before'
        );
        
        
        // Add type="module" to entries-controller
        add_filter('script_loader_tag', function($tag, $handle, $src) {
            if ('cmbsqf-entries-manager' === $handle) {
                // phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedScript -- Modifying enqueued script tag via filter
                $tag = str_replace(' src=', ' type="module" src=', $tag);
            }
            return $tag;
        }, 10, 3);
        
        // Enqueue entries CSS
        wp_enqueue_style(
            'cmbsqf-entries',
            CMBSQF_PLUGIN_URL . 'admin/assets/css/entries.css',
            [],
            CMBSQF_VERSION
        );
    }
    
    /**
     * Main entry point to render the entries list page
     *
     * @param int $form_id Form ID
     */
    public static function render($form_id) {
        // Enqueue assets for modal
        self::enqueue_assets($form_id);
        
        // Get form name
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        $form_name = CMBSQF_DB_Settings::get('form_name', $form_id, 'Form #' . $form_id);
        
        // Create and prepare the table
        $table = new self($form_id);
        $table->prepare_items();
        
        // Build URLs for navigation
        $base_url = admin_url('admin.php?page=' . CMBSQF_Constants::SLUG_FORMS);
        $entries_url = add_query_arg(['view' => 'entries', 'form_id' => $form_id], $base_url);
        $settings_url = add_query_arg(['view' => 'settings', 'form_id' => $form_id], $base_url);
        
        ?>
        <div class="wrap">
            <div class="cmbsqf-header-nav">
                <a href="<?php echo esc_url($base_url); ?>" class="button">
                    &larr; <?php esc_html_e('Back to All Forms', 'cmb-sqlite-form'); ?>
                </a>
                <span class="cmbsqf-header-toggle">
                    [ <strong><?php esc_html_e('Entries', 'cmb-sqlite-form'); ?></strong>
                    <span style="margin: 0 4px;">|</span>
                    <a href="<?php echo esc_url($settings_url); ?>"><?php esc_html_e('Settings', 'cmb-sqlite-form'); ?></a> ]
                </span>
            </div>
            
            <div class="cmbsqf-entries-actions">
                <button type="button" class="button cmbsqf-export-csv" data-form-id="<?php echo (int) $form_id; ?>">
                    <span class="dashicons dashicons-download"></span>
                    <?php esc_html_e('Export CSV', 'cmb-sqlite-form'); ?>
                </button>
                <button type="button" class="button cmbsqf-download-database" data-form-id="<?php echo (int) $form_id; ?>">
                    <span class="dashicons dashicons-database"></span>
                    <?php esc_html_e('Download DB', 'cmb-sqlite-form'); ?>
                </button>
                <button type="button" class="button cmbsqf-clear-all-entries" data-form-id="<?php echo (int) $form_id; ?>" data-form-name="<?php echo esc_attr($form_name); ?>">
                    <span class="dashicons dashicons-trash"></span>
                    <?php esc_html_e('Clear All', 'cmb-sqlite-form'); ?>
                </button>
            </div>

            <h1 class="cmbsqf-header-title">
                <?php esc_html_e('Form Entries', 'cmb-sqlite-form'); ?>
                <span class="cmbsqf-form-badge">
                    <?php echo esc_html($form_name); ?> #<?php echo (int) $form_id; ?>
                </span>
            </h1>
            
            <form method="get">
                <input type="hidden" name="page" value="<?php echo esc_attr(CMBSQF_Constants::SLUG_FORMS); ?>">
                <input type="hidden" name="view" value="entries">
                <input type="hidden" name="form_id" value="<?php echo esc_attr($form_id); ?>">
                
                <?php
                $table->search_box(__('Search Entries', 'cmb-sqlite-form'), 'cmbsqf-entries');
                $table->display();
                ?>
                
                <!-- Per-page selector - Bottom right, compact style -->
                <div class="tablenav bottom">
                    <div class="alignright" style="margin: 10px 0;">
                        <label for="per-page-selector" style="margin-right: 5px; font-size: 13px;"><?php esc_html_e('Show', 'cmb-sqlite-form'); ?></label>
                        <select name="per_page" id="per-page-selector" style="height: 28px; font-size: 13px; vertical-align: middle;" onchange="this.form.submit()">
                            <?php
                            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display parameter for pagination
                            $current_per_page = isset($_REQUEST['per_page']) ? absint($_REQUEST['per_page']) : 25;
                            $options = [10, 25, 50, 100];
                            foreach ($options as $option) {
                                $selected = $current_per_page === $option ? 'selected' : '';
                                printf(
                                    '<option value="%d" %s>%d</option>',
                                    esc_attr($option),
                                    esc_attr($selected),
                                    esc_html($option)
                                );
                            }
                            ?>
                        </select>
                        <span style="margin-left: 5px; font-size: 13px;"><?php esc_html_e('per page', 'cmb-sqlite-form'); ?></span>
                    </div>
                </div>
            </form>
            
            <!-- CSV Export Modal -->
            <div id="cmbsqf-csv-modal" class="cmbsqf-csv-modal-overlay" style="display: none;">
                <div class="cmbsqf-csv-modal">
                    <div class="cmbsqf-csv-modal-header">
                        <h2><?php esc_html_e('Export to CSV', 'cmb-sqlite-form'); ?></h2>
                        <button type="button" class="cmbsqf-csv-modal-close" aria-label="<?php esc_attr_e('Close', 'cmb-sqlite-form'); ?>">
                            <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    
                    <div class="cmbsqf-csv-modal-body">
                        <label class="cmbsqf-csv-modal-label">
                            <?php esc_html_e('Select CSV delimiter:', 'cmb-sqlite-form'); ?>
                        </label>
                        
                        <div class="cmbsqf-csv-option">
                            <label>
                                <input type="radio" name="csv_delimiter" value="comma">
                                <span class="cmbsqf-csv-option-title">
                                    <svg class="cmbsqf-csv-icon" width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <rect x="3" y="3" width="18" height="18" rx="2" stroke="currentColor" stroke-width="2"/>
                                        <line x1="3" y1="9" x2="21" y2="9" stroke="currentColor" stroke-width="2"/>
                                        <line x1="3" y1="15" x2="21" y2="15" stroke="currentColor" stroke-width="2"/>
                                        <line x1="12" y1="3" x2="12" y2="21" stroke="currentColor" stroke-width="2"/>
                                    </svg>
                                    <?php esc_html_e('Comma (,)', 'cmb-sqlite-form'); ?>
                                </span>
                                <span class="cmbsqf-csv-option-hint">
                                    <?php esc_html_e('Countries with decimal point number format', 'cmb-sqlite-form'); ?>
                                </span>
                            </label>
                        </div>
                        
                        <div class="cmbsqf-csv-option">
                            <label>
                                <input type="radio" name="csv_delimiter" value="semicolon">
                                <span class="cmbsqf-csv-option-title">
                                    <svg class="cmbsqf-csv-icon" width="18" height="18" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <rect x="3" y="3" width="18" height="18" rx="2" stroke="currentColor" stroke-width="2"/>
                                        <line x1="3" y1="9" x2="21" y2="9" stroke="currentColor" stroke-width="2"/>
                                        <line x1="3" y1="15" x2="21" y2="15" stroke="currentColor" stroke-width="2"/>
                                        <line x1="12" y1="3" x2="12" y2="21" stroke="currentColor" stroke-width="2"/>
                                    </svg>
                                    <?php esc_html_e('Semicolon (;)', 'cmb-sqlite-form'); ?>
                                </span>
                                <span class="cmbsqf-csv-option-hint">
                                    <?php esc_html_e('Countries with decimal comma number format', 'cmb-sqlite-form'); ?>
                                </span>
                            </label>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}
