<?php

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Pages Controller
 *
 * Purpose: Manages menu registration, page routing, and asset enqueuing
 * Used in context: wp-admin-plugin
 * Location: /admin/class-pages.php
 */
class CMBSQF_Pages {
    /**
     * Register admin menu and asset hooks.
     */
    public static function register() {
        // Load database dependencies
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-settings.php';
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';
        
        add_action('admin_menu', [__CLASS__, 'register_menu_page']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_plugin_assets']);
    }

    /**
     * Enqueue CSS and JavaScript for plugin admin pages.
     */
    public static function enqueue_plugin_assets() {
        $screen = get_current_screen();
        
        // Only enqueue on our plugin pages
        // Check if screen ID contains our plugin prefix or slugs
        /* 
        if (!$screen || (
            strpos($screen->id, CMBSQF_Constants::SLUG_DASHBOARD) === false && 
            strpos($screen->id, CMBSQF_Constants::SLUG_FORMS) === false &&
            strpos($screen->id, CMBSQF_Constants::SLUG_SETTINGS) === false
        )) {
            return;
        }
        */

        // Settings CSS
        wp_enqueue_style(
            'cmbsqf-settings',
            CMBSQF_PLUGIN_URL . 'admin/assets/css/settings.css',
            [],
            CMBSQF_VERSION
        );

        // Settings Controller JavaScript (ES6 Module)
        wp_enqueue_script(
            'cmbsqf-settings-controller',
            CMBSQF_PLUGIN_URL . 'admin/assets/js/settings-controller.js',
            [],
            CMBSQF_VERSION,
            true
        );

        // Localize script for REST API settings
        wp_add_inline_script(
            'cmbsqf-settings-controller',
            sprintf(
                'window.cmbsqfSettings = %s;',
                wp_json_encode([
                    'restUrl' => rest_url('cmbsqf/v1/admin'),
                    'nonce'   => wp_create_nonce('wp_rest'),
                    'downloadDbNonce'  => wp_create_nonce('cmbsqf_download_db'),
                    'downloadCsvNonce' => wp_create_nonce('cmbsqf_download_csv'),
                ])
            ),
            'before'
        );

        // Add type="module" to settings-controller
        add_filter('script_loader_tag', function($tag, $handle, $src) {
            if ('cmbsqf-settings-controller' === $handle) {
                // Use str_replace to preserve inline scripts attached to the tag
                $tag = str_replace(' src=', ' type="module" src=', $tag);
            }
            return $tag;
        }, 10, 3);

        // CodeMirror for CSS editor
        wp_enqueue_code_editor(['type' => 'text/css']);

        // Forms Controller JavaScript (ES6 Module)
        wp_enqueue_script(
            'cmbsqf-forms-manager',
            CMBSQF_PLUGIN_URL . 'admin/assets/js/forms-controller.js',
            ['cmbsqf-settings-controller'], // Depend on settings controller
            CMBSQF_VERSION,
            true
        );

        // Add type="module" to forms-controller
        add_filter('script_loader_tag', function($tag, $handle, $src) {
            if ('cmbsqf-forms-manager' === $handle) {
                // Use str_replace to preserve inline scripts attached to the tag
                $tag = str_replace(' src=', ' type="module" src=', $tag);
            }
            return $tag;
        }, 10, 3);
    }

    /**
     * Register the top-level admin page.
     */
    public static function register_menu_page() {
        add_menu_page(
            __('CMB SQLite Form', 'cmb-sqlite-form'),
            __('CMB SQLite Form', 'cmb-sqlite-form'),
            'manage_options',
            CMBSQF_Constants::SLUG_DASHBOARD,
            [__CLASS__, 'render_dashboard'],
            'dashicons-email'
        );

        add_submenu_page(
            CMBSQF_Constants::SLUG_DASHBOARD,
            __('View All Forms', 'cmb-sqlite-form'),
            __('View All Forms', 'cmb-sqlite-form'),
            'manage_options',
            CMBSQF_Constants::SLUG_FORMS,
            [__CLASS__, 'render_view_all_forms']
        );

        add_submenu_page(
            CMBSQF_Constants::SLUG_DASHBOARD,
            __('Global Settings', 'cmb-sqlite-form'),
            __('Global Settings', 'cmb-sqlite-form'),
            'manage_options',
            CMBSQF_Constants::SLUG_SETTINGS,
            [__CLASS__, 'render_global_settings']
        );

        // Remove the automatic duplicate submenu that mirrors the parent.
        remove_submenu_page(CMBSQF_Constants::SLUG_DASHBOARD, CMBSQF_Constants::SLUG_DASHBOARD);
    }

    /**
     * Render the dashboard placeholder page.
     */
    public static function render_dashboard() {
        //self::render_admin_layout();
    }

    /**
     * Render the "View All Forms" submenu.
     */
    public static function render_view_all_forms() {
        $view = sanitize_key(filter_input(INPUT_GET, 'view', FILTER_DEFAULT));

        switch ($view) {
            case 'edit':
                // TODO: Render form editor (view and edit fields, layout, etc.)
                break;

            case 'entries':
                $form_id = filter_input(INPUT_GET, 'form_id', FILTER_VALIDATE_INT);
                if ($form_id > 0) {
                    require_once CMBSQF_PLUGIN_DIR . 'admin/pages/class-page-form-entries.php';
                    CMBSQF_Page_Form_Entries::render($form_id);
                    return;
                }
                break;

            case 'settings':
                $form_id = filter_input(INPUT_GET, 'form_id', FILTER_VALIDATE_INT);
                if ($form_id > 0) {
                    require_once CMBSQF_PLUGIN_DIR . 'admin/pages/class-page-form-settings.php';
                    CMBSQF_Page_Form_Settings::render($form_id);
                    return;
                }
                break;

            default:
                require_once CMBSQF_PLUGIN_DIR . 'admin/pages/class-page-forms-all.php';
                CMBSQF_Page_Forms::render();
                break;
        }
    }

    /**
     * Render the "Global Settings" submenu.
     */
    public static function render_global_settings() {
        require_once CMBSQF_PLUGIN_DIR . 'admin/pages/class-page-global-settings.php';
        CMBSQF_Page_Global_Settings::render();
    }
}
