/**
 * Unsaved Changes Warning Module
 * 
 * Tracks form changes and warns users before navigating away
 */

export default class UnsavedChanges {
    constructor() {
        this.forms = document.querySelectorAll('.cmbsqf-settings-form');
        this.hasUnsavedChanges = false;
        this.init();
    }

    init() {
        if (!this.forms.length) return;

        // Track changes in all forms
        this.forms.forEach(form => {
            this.trackFormChanges(form);
            this.handleFormSubmit(form);
        });

        // Warn before tab navigation
        this.warnOnTabChange();

        // Warn before leaving page
        this.warnOnPageLeave();
    }

    /**
     * Track changes in a form
     */
    trackFormChanges(form) {
        const inputs = form.querySelectorAll('input, textarea, select');

        inputs.forEach(input => {
            // Store original value
            input.dataset.originalValue = this.getInputValue(input);

            // Listen for changes
            input.addEventListener('input', () => {
                this.checkForChanges(form);
            });

            input.addEventListener('change', () => {
                this.checkForChanges(form);
            });
        });
    }

    /**
     * Check if form has unsaved changes
     */
    checkForChanges(form) {
        const inputs = form.querySelectorAll('input, textarea, select');
        let hasChanges = false;

        inputs.forEach(input => {
            const originalValue = input.dataset.originalValue || '';
            const currentValue = this.getInputValue(input);

            if (originalValue !== currentValue) {
                hasChanges = true;
            }
        });

        // Mark form as dirty
        if (hasChanges) {
            form.classList.add('has-unsaved-changes');
            this.hasUnsavedChanges = true;
        } else {
            form.classList.remove('has-unsaved-changes');
            this.updateGlobalState();
        }
    }

    /**
     * Get input value (handles different input types)
     */
    getInputValue(input) {
        if (input.type === 'checkbox') {
            return input.checked ? '1' : '0';
        } else if (input.type === 'radio') {
            return input.checked ? input.value : '';
        } else {
            return input.value;
        }
    }

    /**
     * Update global unsaved changes state
     */
    updateGlobalState() {
        const anyFormDirty = document.querySelector('.cmbsqf-settings-form.has-unsaved-changes');
        this.hasUnsavedChanges = !!anyFormDirty;
    }

    /**
     * Reset form tracking after successful save
     */
    handleFormSubmit(form) {
        form.addEventListener('submit', (e) => {
            // Remove dirty flag when submitting
            form.classList.remove('has-unsaved-changes');

            // Wait for successful save to reset original values
            // This will be handled by the AJAX success callback
        });
    }

    /**
     * Reset all inputs to current values (after successful save)
     */
    resetForm(form) {
        const inputs = form.querySelectorAll('input, textarea, select');

        inputs.forEach(input => {
            input.dataset.originalValue = this.getInputValue(input);
        });

        form.classList.remove('has-unsaved-changes');
        this.updateGlobalState();
    }

    /**
     * Warn before navigating to another tab
     */
    warnOnTabChange() {
        const tabButtons = document.querySelectorAll('.nav-tab');

        tabButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                if (this.hasUnsavedChanges) {
                    const confirmed = confirm(
                        'You have unsaved changes. If you leave this page, your changes will be lost.\n\n' +
                        'Do you want to leave without saving?'
                    );

                    if (!confirmed) {
                        e.preventDefault();
                        e.stopPropagation();
                        return false;
                    }
                }
            });
        });

        // Also warn on sidebar section clicks (if applicable)
        const sidebarLinks = document.querySelectorAll('.cmbsqf-sidebar-nav a');

        sidebarLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                if (this.hasUnsavedChanges) {
                    const confirmed = confirm(
                        'You have unsaved changes. If you leave this section, your changes will be lost.\n\n' +
                        'Do you want to leave without saving?'
                    );

                    if (!confirmed) {
                        e.preventDefault();
                        e.stopPropagation();
                        return false;
                    }
                }
            });
        });
    }

    /**
     * Warn before leaving page (browser navigation)
     */
    warnOnPageLeave() {
        window.addEventListener('beforeunload', (e) => {
            if (this.hasUnsavedChanges) {
                e.preventDefault();
                // Modern browsers ignore custom messages and show their own
                e.returnValue = '';
                return '';
            }
        });
    }

    /**
     * Public method to reset a specific form (called after AJAX save)
     */
    static resetFormById(formElement) {
        const instance = window.cmbsqfUnsavedChanges;
        if (instance && formElement) {
            instance.resetForm(formElement);
        }
    }
}
