/**
 * Settings UI Module
 * Handles generic DOM interactions (Notices, Toggles).
 */

export const UI = {
    /**
     * Show admin notice.
     * @param {string} type 'success' or 'error'
     * @param {string} message Notice message
     */
    showNotice(type, message) {
        const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
        const notice = document.createElement('div');
        notice.className = `notice ${noticeClass} is-dismissible`;
        notice.innerHTML = `<p>${this.escapeHtml(message)}</p>`;

        const dismissButton = document.createElement('button');
        dismissButton.type = 'button';
        dismissButton.className = 'notice-dismiss';
        dismissButton.innerHTML = '<span class="screen-reader-text">Dismiss this notice.</span>';
        dismissButton.addEventListener('click', () => notice.remove());
        notice.appendChild(dismissButton);

        const wrap = document.querySelector('.wrap');
        if (wrap) wrap.insertBefore(notice, wrap.firstChild);

        setTimeout(() => {
            if (notice.parentNode) {
                notice.style.transition = 'opacity 0.3s ease';
                notice.style.opacity = '0';
                setTimeout(() => notice.remove(), 300);
            }
        }, type === 'success' ? 5000 : 8000);
    },

    /**
     * Remove all existing notices.
     */
    removeNotices() {
        document.querySelectorAll('.notice.is-dismissible').forEach(n => n.remove());
    },

    /**
     * Escape HTML to prevent XSS.
     * @param {string} text Text to escape
     * @returns {string} Escaped text
     */
    escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    },

    /**
     * Initialize toggle controls.
     */
    initToggles() {
        const controls = document.querySelectorAll('[data-toggle-group]');

        controls.forEach(control => {
            const group = control.getAttribute('data-toggle-group');
            const targets = document.querySelectorAll(`[data-toggle-target="${group}"]`);

            const updateVisibility = () => {
                let shouldShow = false;
                if (control.type === 'checkbox') {
                    shouldShow = control.checked;
                } else if (control.type === 'radio') {
                    const showValue = control.getAttribute('data-toggle-value') || '1';
                    shouldShow = control.checked && control.value === showValue;
                }
                targets.forEach(t => {
                    const displayType = t.tagName === 'TR' ? 'table-row' : 'block';
                    t.style.display = shouldShow ? displayType : 'none';
                });
            };

            control.addEventListener('change', updateVisibility);

            if (control.type === 'radio') {
                const radios = document.querySelectorAll(`input[name="${control.getAttribute('name')}"]`);
                radios.forEach(r => r.addEventListener('change', updateVisibility));
            }

            updateVisibility();
        });
    }
};
