/**
 * Section: Styling (CodeMirror & Templates)
 * Handles CSS editor and template loading.
 */

let cssCodeMirror = null;

/**
 * Initialize CodeMirror for CSS textarea.
 */
function initCodeMirror() {
    const cssEditor = document.getElementById('form__styling__custom_css');

    if (!cssEditor || typeof wp === 'undefined' || typeof wp.codeEditor === 'undefined') {
        return;
    }

    const editorSettings = wp.codeEditor.defaultSettings ? Object.assign({}, wp.codeEditor.defaultSettings) : {};
    editorSettings.codemirror = Object.assign({}, editorSettings.codemirror, {
        indentUnit: 4,
        tabSize: 4,
        indentWithTabs: true,
        lineNumbers: true,
        mode: 'css',
        lineWrapping: true,
        styleActiveLine: true,
        matchBrackets: true,
        autoCloseBrackets: true,
    });

    cssCodeMirror = wp.codeEditor.initialize(cssEditor, editorSettings);

    // Refresh CodeMirror when section becomes visible (fixes rendering in hidden tabs)
    document.addEventListener('cmbsqf:section-shown', function (e) {
        if (e.detail.sectionId === 'styling' && cssCodeMirror && cssCodeMirror.codemirror) {
            cssCodeMirror.codemirror.refresh();
        }
    });
}

/**
 * Initialize template CSS loader.
 */
function initTemplateLoader() {
    const templateSelect = document.getElementById('cmb_sf_template_select');
    const loadBtn = document.getElementById('cmb_sf_load_template_btn');
    const cssEditor = document.getElementById('form__styling__custom_css');

    if (!templateSelect || !loadBtn || !cssEditor) {
        return;
    }

    loadBtn.addEventListener('click', function () {
        const selectedTemplate = templateSelect.value;

        if (!selectedTemplate) {
            alert('Please select a template first.');
            return;
        }

        const selectedOption = templateSelect.options[templateSelect.selectedIndex];
        const css = selectedOption.getAttribute('data-css');

        if (css) {
            // Warn user about overwriting current content
            if (!confirm('Loading this template will replace your current CSS. Continue?')) {
                return;
            }

            if (cssCodeMirror && cssCodeMirror.codemirror) {
                cssCodeMirror.codemirror.setValue(css);
            } else {
                cssEditor.value = css;
            }
            templateSelect.value = '';
        } else {
            alert('Template CSS not found.');
        }
    });
}

/**
 * Save CodeMirror content to textarea.
 * Must be called before form submission.
 */
export function saveStyling() {
    if (cssCodeMirror && cssCodeMirror.codemirror) {
        cssCodeMirror.codemirror.save();
    }
}

export default function initStyling() {
    initCodeMirror();
    initTemplateLoader();
}
