/**
 * Section: Junk Email Blocklist Management
 * Handles dynamic source management and blocklist updates.
 */

let metadataCache = {};

/**
 * Validate a single source URL
 */
function validateSource(url, itemElement, forceShow = false) {
    let statusDiv = itemElement.querySelector('.cmbsqf-source-status');
    if (!statusDiv) {
        statusDiv = document.createElement('div');
        statusDiv.className = 'cmbsqf-source-status';
        itemElement.appendChild(statusDiv);
    }

    if (forceShow) {
        statusDiv.style.display = 'block';
        const infoBtn = itemElement.querySelector('.cmbsqf-source-info');
        if (infoBtn) infoBtn.classList.add('active');
    }

    statusDiv.className = 'cmbsqf-source-status cmbsqf-status-pending';
    statusDiv.textContent = '⏳ Validating...';

    fetch(window.cmbsqfSettings.restUrl + '/validate-source', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': window.cmbsqfSettings.nonce
        },
        body: JSON.stringify({ url: url })
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                statusDiv.className = 'cmbsqf-source-status cmbsqf-status-success';
                statusDiv.textContent = `✓ Valid source (${Number(data.domains).toLocaleString()} domains found)`;

                metadataCache[url] = {
                    url: url,
                    status: 'success',
                    domains: data.domains,
                    last_updated: 'Just now'
                };
            } else {
                statusDiv.className = 'cmbsqf-source-status cmbsqf-status-error';
                statusDiv.textContent = `⚠ ${data.message}`;
            }
        })
        .catch(() => {
            statusDiv.className = 'cmbsqf-source-status cmbsqf-status-error';
            statusDiv.textContent = '⚠ Validation failed';
        });
}

function toggleSourceStatus(url, itemElement) {
    let statusDiv = itemElement.querySelector('.cmbsqf-source-status');
    let infoBtn = itemElement.querySelector('.cmbsqf-source-info');

    if (!statusDiv) {
        validateSource(url, itemElement, true);
        return;
    }

    if (statusDiv.style.display === 'none') {
        statusDiv.style.display = 'block';
        if (infoBtn) infoBtn.classList.add('active');
        validateSource(url, itemElement, true);
    } else {
        statusDiv.style.display = 'none';
        if (infoBtn) infoBtn.classList.remove('active');
    }
}

function escapeHtml(text) {
    if (!text) return '';
    return text
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}

function addSource() {
    const input = document.getElementById('cmbsqf-new-source-url');
    const url = input.value.trim();

    if (!url) {
        alert('Please enter a valid URL');
        return;
    }

    try {
        new URL(url);
    } catch (_) {
        alert('Invalid URL format');
        return;
    }

    const list = document.getElementById('cmbsqf-sources-list');
    const existing = list.querySelector(`input[value="${CSS.escape(url)}"]`);
    if (existing) {
        alert('This source is already in the list');
        input.value = '';
        return;
    }

    const div = document.createElement('div');
    div.className = 'cmbsqf-source-item';
    div.setAttribute('data-url', url);

    div.innerHTML = `
        <code class="cmbsqf-source-url">${escapeHtml(url)}</code>
        <div class="cmbsqf-source-actions">
            <button type="button" class="button button-small cmbsqf-source-info active" title="View Info">
                <span class="dashicons dashicons-info"></span>
            </button>
            <button type="button" class="button button-small cmbsqf-remove-source" title="Remove">
                <span class="dashicons dashicons-trash"></span>
            </button>
        </div>
        <input type="hidden" name="form__junk_email__sources[]" value="${escapeHtml(url)}">
    `;

    list.appendChild(div);
    input.value = '';

    validateSource(url, div, true);
}

function updateCombinedStats(totalDomains) {
    const combinedP = document.querySelector('.cmbsqf-blocklist-combined p');
    if (combinedP) {
        combinedP.innerHTML =
            'Total unique domains: <strong>' + Number(totalDomains).toLocaleString() + '</strong><br>' +
            'Last updated: <strong>Just now</strong>';
    }
}

function updateBlocklist(button) {
    const formId = button.getAttribute('data-form-id') || 0;
    const status = document.querySelector('.cmbsqf-blocklist-status');
    const list = document.getElementById('cmbsqf-sources-list');

    const sources = [];
    const inputs = list.querySelectorAll('input[name="form__junk_email__sources[]"]');

    inputs.forEach(function (input) {
        const url = input.value.trim();
        if (url) sources.push(url);
    });

    button.disabled = true;
    const icon = button.querySelector('.dashicons');
    if (icon) icon.classList.add('spin');
    status.innerHTML = '<p style="color: #666;">Downloading and processing sources... This may take a few moments.</p>';

    fetch(window.cmbsqfSettings.restUrl + '/update-blocklist', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': window.cmbsqfSettings.nonce
        },
        body: JSON.stringify({
            form_id: formId,
            sources: sources
        })
    })
        .then(response => {
            if (!response.ok) {
                return response.json().then(err => { throw new Error(err.message || 'Update failed'); });
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                if (data.sources_metadata) {
                    data.sources_metadata.forEach(item => {
                        if (item.url) metadataCache[item.url] = item;
                    });
                }

                updateCombinedStats(data.total_domains);
                status.innerHTML = `<p style="color: #46b450;">✓ ${data.message}</p>`;

                const infoPanel = document.getElementById('cmbsqf-source-info-panel');
                if (infoPanel) infoPanel.style.display = 'none';

                setTimeout(() => {
                    status.innerHTML = '';
                }, 5000);
            } else {
                status.innerHTML = `<p style="color: #dc3232;">✕ ${data.message || 'Update failed.'}</p>`;
            }
        })
        .catch(error => {
            status.innerHTML = `<p style="color: #dc3232;">✕ ${error.message}</p>`;
        })
        .finally(() => {
            button.disabled = false;
            if (icon) icon.classList.remove('spin');
        });
}

export default function initJunkEmail() {
    const listContainer = document.getElementById('cmbsqf-sources-list');
    if (!listContainer) return;

    // Load initial metadata
    try {
        const rawMeta = listContainer.getAttribute('data-metadata');
        if (rawMeta) {
            const metaArray = JSON.parse(rawMeta);
            metaArray.forEach(item => {
                if (item.url) metadataCache[item.url] = item;
            });
        }
    } catch (e) {
        console.error('Failed to parse blocklist metadata', e);
    }

    // Add source button
    const addBtn = document.getElementById('cmbsqf-add-source-btn');
    if (addBtn) {
        addBtn.addEventListener('click', addSource);
    }

    // Add source on Enter key
    const addInput = document.getElementById('cmbsqf-new-source-url');
    if (addInput) {
        addInput.addEventListener('keypress', function (e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                addSource();
            }
        });
    }

    // List actions (delegated)
    listContainer.addEventListener('click', function (e) {
        const target = e.target;

        if (target.closest('.cmbsqf-remove-source')) {
            target.closest('.cmbsqf-source-item').remove();
        }

        if (target.closest('.cmbsqf-source-info')) {
            const item = target.closest('.cmbsqf-source-item');
            const url = item.getAttribute('data-url');
            if (url) {
                toggleSourceStatus(url, item);
            }
        }
    });

    // Update blocklist button
    const updateBtn = document.querySelector('.cmbsqf-update-blocklist');
    if (updateBtn) {
        updateBtn.addEventListener('click', function () {
            updateBlocklist(this);
        });
    }

    // Add spin animation for dashicons
    const style = document.createElement('style');
    style.textContent = `
        .dashicons.spin {
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
    `;
    document.head.appendChild(style);
}
