/**
 * Section: Outgoing Server (Test Email)
 * Handles test email functionality.
 */

import { API } from './rest-api.js';
import { UI } from './ui.js';

function renderTestEmailResult(data, isSuccess) {
    let html = '';

    if (isSuccess) {
        html += '<div class="cmbsqf-test-email-success">';
        html += '<strong>✓ Test Passed</strong>';
        html += `<p>${UI.escapeHtml(data.message)}</p>`;
        html += '</div>';
    } else {
        html += '<div class="cmbsqf-test-email-error">';
        html += '<strong>✕ Test Failed</strong>';
        html += `<p>${UI.escapeHtml(data.message)}`;
        if (data.error_details) {
            html += `<br><span class="cmbsqf-test-email-error-details">Error: ${UI.escapeHtml(data.error_details.message)}</span>`;
        }
        html += '</p>';
        html += '</div>';
    }

    if (data.debug_log && data.debug_log.trim()) {
        html += '<div class="cmbsqf-test-email-debug-wrapper">';
        html += '<details class="cmbsqf-examples-details">';
        html += '<summary class="cmbsqf-examples-summary"><strong>Debug Output</strong></summary>';
        html += '<div class="cmbsqf-examples-content">';
        html += `<pre class="cmbsqf-test-email-debug-log">${UI.escapeHtml(data.debug_log)}</pre>`;
        html += '<button type="button" class="button button-small cmbsqf-test-email-copy-btn" onclick="navigator.clipboard.writeText(this.previousElementSibling.textContent).then(() => alert(\'Debug log copied to clipboard\'))">Copy Log</button>';
        html += '</div>';
        html += '</details>';
        html += '</div>';
    }

    return html;
}

export default function initEmail() {
    const sendButton = document.getElementById('cmbsqf_send_test_email');
    const recipientInput = document.getElementById('cmbsqf_test_email_recipient');
    const spinner = document.querySelector('#cmbsqf_send_test_email + .spinner');
    const resultDiv = document.getElementById('cmbsqf_test_email_result');

    if (!sendButton || !recipientInput) {
        return;
    }

    sendButton.addEventListener('click', async function () {
        const recipient = recipientInput.value.trim();

        if (!recipient || !recipient.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/)) {
            alert('Please enter a valid email address.');
            recipientInput.focus();
            return;
        }

        const smtpEnabled = document.querySelector('input[name="notifications__outgoing_server__smtp_enabled"]:checked')?.value || '0';
        const smtpHost = document.getElementById('notifications__outgoing_server__smtp_host')?.value || '';
        const smtpPort = document.getElementById('notifications__outgoing_server__smtp_port')?.value || '587';
        const smtpUsername = document.getElementById('notifications__outgoing_server__smtp_username')?.value || '';
        const smtpPassword = document.getElementById('notifications__outgoing_server__smtp_password')?.value || '';
        const smtpEncryption = document.getElementById('notifications__outgoing_server__smtp_encryption')?.value || 'tls';
        const fromName = document.getElementById('notifications__sender__from_name')?.value || '';
        const fromEmail = document.getElementById('notifications__sender__from_email')?.value || '';

        sendButton.disabled = true;
        if (spinner) spinner.classList.add('is-active');
        resultDiv.style.display = 'none';

        try {
            const response = await API.sendTestEmail({
                recipient,
                smtp_enabled: smtpEnabled,
                smtp_host: smtpHost,
                smtp_port: parseInt(smtpPort, 10),
                smtp_username: smtpUsername,
                smtp_password: smtpPassword,
                smtp_encryption: smtpEncryption,
                from_name: fromName,
                from_email: fromEmail,
            });

            const isSuccess = response.success === true;
            resultDiv.style.display = 'block';
            resultDiv.className = '';
            resultDiv.style.padding = '0';
            resultDiv.style.background = 'transparent';
            resultDiv.style.border = 'none';
            resultDiv.innerHTML = renderTestEmailResult(response, isSuccess);

        } catch (error) {
            console.error('Error sending test email:', error);
            resultDiv.style.display = 'block';
            resultDiv.className = 'notice notice-error inline';
            resultDiv.innerHTML = `<p><strong>Error:</strong> ${UI.escapeHtml(error.message)}</p>`;
        } finally {
            sendButton.disabled = false;
            if (spinner) spinner.classList.remove('is-active');
        }
    });
}
