/**
 * Settings API Module
 * Handles all REST API communication.
 */


export const API = {
    /**
     * Save settings to server.
     * @param {string} context 'global' or 'form'
     * @param {number|null} formId Form ID (required if context is 'form')
     * @param {Object} settings Settings key-value pairs
     * @returns {Promise<Object>} Response data
     */
    async saveSettings(context, formId, settings) {
        if (!['global', 'form'].includes(context)) {
            throw new Error('Invalid context. Must be "global" or "form".');
        }

        if (context === 'form' && (!formId || formId <= 0)) {
            throw new Error('Form ID is required for form context.');
        }

        const url = `${window.cmbsqfSettings.restUrl}/settings`;

        try {
            const body = { context, settings };
            if (formId) body.form_id = formId;

            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': window.cmbsqfSettings.nonce,
                },
                body: JSON.stringify(body),
            });

            if (!response.ok) {
                const error = await response.json();
                throw new Error(error.message || 'Failed to save settings.');
            }

            return await response.json();
        } catch (error) {
            console.error('Error saving settings:', error);
            throw error;
        }
    },

    /**
     * Send test email.
     * @param {Object} params Email parameters
     * @returns {Promise<Object>} Response data
     */
    async sendTestEmail(params) {
        const url = `${window.cmbsqfSettings.restUrl}/test-email`;

        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': window.cmbsqfSettings.nonce,
                },
                body: JSON.stringify(params),
            });

            return await response.json();
        } catch (error) {
            console.error('Error sending test email:', error);
            throw error;
        }
    }
};
