/**
 * Settings Controller Module (Entry Point)
 * Orchestrates form submission and initialization.
 */

import { API } from './settings/rest-api.js';
import { UI } from './settings/ui.js';
import initStyling, { saveStyling } from './settings/section-styling.js';
import initEmail from './settings/section-email.js';
import initTabNavigation from './settings/tabs.js';
import initJunkEmail from './settings/section-junk-email.js';
import UnsavedChanges from './settings/unsaved-changes.js';

/**
 * Collect form data including unchecked checkboxes.
 * @param {HTMLFormElement} form Form element
 * @returns {Object} Form data as key-value pairs
 */
function collectFormData(form) {
	const formData = new FormData(form);
	const settings = {};

	for (const [key, value] of formData.entries()) {
		if (key.endsWith('[]')) {
			settings[key] = formData.getAll(key);
		} else {
			settings[key] = value;
		}
	}

	// Handle unchecked checkboxes
	form.querySelectorAll('input[type="checkbox"]').forEach(checkbox => {
		if (!checkbox.checked && checkbox.name) {
			settings[checkbox.name] = '0';
		}
	});

	return settings;
}

/**
 * Initialize form submission handler.
 */
function initFormSubmit() {
	const forms = document.querySelectorAll('.cmbsqf-settings-form');

	forms.forEach(form => {
		form.addEventListener('submit', async function (e) {
			e.preventDefault();

			const context = form.getAttribute('data-context') || 'global';
			const formId = form.getAttribute('data-form-id') ? parseInt(form.getAttribute('data-form-id'), 10) : null;
			const submitButton = form.querySelector('button[type="submit"], input[type="submit"]');
			const originalText = submitButton ? submitButton.textContent : '';

			if (submitButton) {
				submitButton.disabled = true;
				submitButton.textContent = 'Saving...';
			}

			UI.removeNotices();

			try {
				// Sync CodeMirror content to textarea
				saveStyling();

				const settings = collectFormData(form);
				const response = await API.saveSettings(context, formId, settings);

				UI.showNotice('success', response.message || 'Settings saved successfully.');
				window.scrollTo({ top: 0, behavior: 'smooth' });

				// Reset unsaved changes tracking
				UnsavedChanges.resetFormById(form);
			} catch (error) {
				UI.showNotice('error', error.message || 'Failed to save settings. Please try again.');
			} finally {
				if (submitButton) {
					submitButton.disabled = false;
					submitButton.textContent = originalText;
				}
			}
		});
	});
}

// Initialize when DOM is ready
function init() {
	initTabNavigation();
	UI.initToggles();
	initFormSubmit();
	initStyling();
	initEmail();
	initJunkEmail();

	// Initialize unsaved changes warning
	window.cmbsqfUnsavedChanges = new UnsavedChanges();
}

if (document.readyState === 'loading') {
	document.addEventListener('DOMContentLoaded', init);
} else {
	init();
}
