/**
 * Forms CRUD Operations
 * Handle create, rename, and delete operations for forms
 */

import { fetchWithNonce, getRestUrl, FADE_OUT_DURATION, TABLE_CHECK_DELAY } from './forms-utils.js';

/**
 * Handle form creation
 */
export async function handleCreateForm(e) {
    e.preventDefault();

    const formName = prompt('Enter a name for the new form:');
    if (!formName || formName.trim() === '') {
        return;
    }

    try {
        const response = await fetchWithNonce(getRestUrl('/forms'), {
            method: 'POST',
            body: JSON.stringify({ name: formName.trim() }),
        });

        const data = await response.json();

        if (response.ok && data.success) {
            // Reload page to show new form
            window.location.reload();
        } else {
            alert('Error: ' + (data.message || 'Failed to create form'));
        }
    } catch (error) {
        console.error('Error creating form:', error);
        alert('Error creating form. Please try again.');
    }
}

/**
 * Handle form renaming
 */
export async function handleRenameForm(trigger) {
    const formId = parseInt(trigger.getAttribute('data-form-id'));
    const row = trigger.closest('tr');
    const nameCell = row ? row.querySelector('.column-form_name strong') : null;
    const currentName = nameCell ? nameCell.textContent.trim() : '';

    const newName = prompt('Enter new name for the form:', currentName);
    if (!newName || newName.trim() === '' || newName.trim() === currentName) {
        return;
    }

    try {
        const response = await fetchWithNonce(getRestUrl(`/forms/${formId}`), {
            method: 'PATCH',
            body: JSON.stringify({ name: newName.trim() }),
        });

        const data = await response.json();

        if (response.ok && data.success) {
            // Update DOM
            if (nameCell) {
                nameCell.textContent = newName.trim();
            }
        } else {
            alert('Error: ' + (data.message || 'Failed to rename form'));
        }
    } catch (error) {
        console.error('Error renaming form:', error);
        alert('Error renaming form. Please try again.');
    }
}

/**
 * Handle form deletion
 */
export async function handleDeleteForm(trigger) {
    const formId = parseInt(trigger.getAttribute('data-form-id'));
    const row = trigger.closest('tr');
    const formName = row ? row.querySelector('.column-form_name strong')?.textContent.trim() : `Form #${formId}`;

    if (!confirm(`Are you sure you want to delete "${formName}"?\n\nThis will also delete all entries and settings associated with this form. This action cannot be undone.`)) {
        return;
    }

    try {
        const response = await fetchWithNonce(getRestUrl(`/forms/${formId}`), {
            method: 'DELETE',
        });

        const data = await response.json();

        if (response.ok && data.success) {
            // Fade out row
            if (row) {
                row.style.backgroundColor = '#ffcccc';
                row.style.transition = `background-color ${FADE_OUT_DURATION}ms`;
                setTimeout(() => row.remove(), FADE_OUT_DURATION);
            }

            // Check if table is empty
            setTimeout(() => {
                const tbody = document.querySelector('.wp-list-table tbody');
                if (tbody && tbody.children.length === 0) {
                    window.location.reload();
                }
            }, TABLE_CHECK_DELAY);
        } else {
            alert('Error: ' + (data.message || 'Failed to delete form'));
        }
    } catch (error) {
        console.error('Error deleting form:', error);
        alert('Error deleting form. Please try again.');
    }
}

/**
 * Handle bulk delete action
 */
export async function handleBulkDelete(ids) {
    if (!confirm(`Are you sure you want to delete ${ids.length} selected forms?\n\nThis will also delete all entries and settings associated with these forms. This action cannot be undone.`)) {
        return;
    }

    let successCount = 0;
    for (const id of ids) {
        try {
            const response = await fetchWithNonce(getRestUrl(`/forms/${id}`), {
                method: 'DELETE',
            });

            const data = await response.json();

            if (response.ok && data.success) {
                successCount++;
                // Fade out row
                const row = document.querySelector(`tr[data-form-id="${id}"]`);
                if (row) {
                    row.style.backgroundColor = '#ffcccc';
                    row.style.transition = `background-color ${FADE_OUT_DURATION}ms`;
                    setTimeout(() => row.remove(), FADE_OUT_DURATION);
                }
            } else {
                console.error('Failed to delete form:', id, data.message);
            }
        } catch (error) {
            console.error('Error deleting form:', id, error);
        }
    }

    // Check if table is empty after all deletions
    setTimeout(() => {
        const tbody = document.querySelector('.wp-list-table tbody');
        if (tbody && tbody.children.length === 0) {
            window.location.reload();
        }
    }, TABLE_CHECK_DELAY);
}
