/**
 * Entries Utilities
 * Shared utility functions for entries management
 */

/**
 * Format date to YYYY-MM-DD HH:MM:SS
 */
export function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    const seconds = String(date.getSeconds()).padStart(2, '0');
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
}

/**
 * Escape HTML to prevent XSS
 */
export function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Validate email address
 */
export function isValidEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
}

/**
 * Show toast notification
 */
export function showToast(message, type = 'info') {
    const existingToast = document.querySelector('.cmbsqf-toast');
    if (existingToast) {
        existingToast.remove();
    }

    const toast = document.createElement('div');
    toast.className = `cmbsqf-toast cmbsqf-toast-${type}`;
    toast.textContent = message;

    document.body.appendChild(toast);

    setTimeout(() => {
        toast.classList.add('cmbsqf-toast-show');
    }, 10);

    setTimeout(() => {
        toast.classList.remove('cmbsqf-toast-show');
        setTimeout(() => {
            toast.remove();
        }, 300);
    }, 3000);
}

/**
 * Get email from table row by entry ID
 */
export function getEmailFromRow(entryId) {
    const row = document.querySelector(`tr[data-entry-id="${entryId}"]`);
    const emailCell = row ? row.querySelector('.column-email') : null;
    return emailCell ? emailCell.textContent.trim() : 'Unknown';
}

/**
 * Fetch wrapper with automatic nonce injection
 */
export async function fetchWithNonce(url, options = {}) {
    const defaultOptions = {
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': window.cmbsqfSettings.nonce,
            ...(options.headers || {})
        }
    };

    return fetch(url, { ...options, ...defaultOptions });
}

/**
 * Get REST API base URL
 */
export function getRestUrl(endpoint = '') {
    return `${window.cmbsqfSettings.restUrl}${endpoint}`;
}

/**
 * Download database file for a form
 */
export function downloadDatabase(formId) {
    try {
        // Construct admin-post.php URL with nonce
        const url = new URL(window.location.origin + '/wp-admin/admin-post.php');
        url.searchParams.set('action', 'cmbsqf_download_db');
        url.searchParams.set('form_id', formId);
        url.searchParams.set('_wpnonce', window.cmbsqfSettings.downloadDbNonce);

        // Trigger download via navigation (browser handles download automatically)
        window.location.href = url.toString();
    } catch (error) {
        showToast('Failed to download database', 'error');
        console.error('Download error:', error);
    }
}

/**
 * Clear all entries for a form
 */
export async function clearAllEntries(formId, formName) {
    // Show confirmation dialog
    const message = `Are you sure you want to delete ALL entries from "${formName}"?\n\nThis action cannot be undone and will permanently delete all ${formName} submissions.`;

    if (!confirm(message)) {
        return;
    }

    try {
        // Construct clear URL
        const url = getRestUrl(`/forms/${formId}/clear-entries`);

        // Make POST request
        const response = await fetchWithNonce(url, {
            method: 'POST'
        });

        const data = await response.json();

        // Check if response is ok
        if (!response.ok || !data.success) {
            throw new Error(data.message || 'Failed to clear entries');
        }

        // Show success message
        showToast(data.message || 'All entries cleared successfully', 'success');

        // Apply fade-out animation to all rows (same as individual delete)
        const rows = document.querySelectorAll('.wp-list-table tbody tr');
        rows.forEach(row => {
            row.style.backgroundColor = '#ffcccc';
        });

        // Reload page after animation
        setTimeout(() => {
            window.location.reload();
        }, 500);

    } catch (error) {
        showToast(error.message || 'Failed to clear entries', 'error');
        console.error('Clear all error:', error);
    }
}

