/**
 * Entries Modal
 * Individual entry modal viewing, editing, and actions
 */

import { fetchWithNonce, getRestUrl, formatDate, escapeHtml, isValidEmail, showToast } from './entries-utils.js';

let currentEntry = null;
let isEditMode = false;

/**
 * Open modal for entry
 */
export async function openModal(entryId) {
    try {
        const response = await fetchWithNonce(getRestUrl(`/entries/${entryId}?form_id=${window.cmbsqfSettings.formId}`), {
            method: 'GET',
        });

        const data = await response.json();

        if (response.ok && data.success) {
            currentEntry = data.data;
            isEditMode = false;
            renderModal(currentEntry);
        } else {
            console.error('Failed to fetch entry:', data.message);
            alert('Error loading entry details.');
        }
    } catch (error) {
        console.error('Error fetching entry:', error);
        alert('Error loading entry details.');
    }
}

/**
 * Close modal
 */
export function closeModal() {
    const overlay = document.querySelector('.cmbsqf-modal-overlay');
    if (overlay) {
        overlay.remove();
        document.body.style.overflow = '';
        currentEntry = null;
        isEditMode = false;
    }
}

/**
 * Render modal HTML
 */
function renderModal(entry) {
    const overlay = document.createElement('div');
    overlay.className = 'cmbsqf-modal-overlay';

    const deliveryStatus = getDeliveryStatusHTML(entry);

    overlay.innerHTML = `
        <div class="cmbsqf-modal-container">
            <div class="cmbsqf-modal-header">
                <h2>Entry #${entry.id}</h2>
                <button type="button" class="cmbsqf-modal-close" aria-label="Close">
                    <span class="dashicons dashicons-no-alt"></span>
                </button>
            </div>
            <div class="cmbsqf-modal-body">
                <div class="cmbsqf-entry-meta">
                    <p><span class="dashicons dashicons-calendar-alt"></span> ${formatDate(entry.created_at)}</p>
                    <p><span class="dashicons dashicons-admin-site-alt3"></span> IP: ${escapeHtml(entry.ip_address || 'N/A')}</p>
                    <p><span class="dashicons dashicons-desktop"></span> ${escapeHtml(entry.user_agent || 'N/A')}</p>
                </div>
                
                <div class="cmbsqf-entry-divider"></div>
                
                <div class="cmbsqf-entry-fields">
                    ${renderField('Name', 'name', entry.name)}
                    ${renderField('Surname', 'surname', entry.surname)}
                    ${renderField('Email', 'email', entry.email, 'email')}
                    ${renderField('Phone', 'phone', entry.phone, 'tel')}
                    ${renderField('Organization', 'organization', entry.organization)}
                    ${renderField('Position', 'position', entry.position)}
                    ${renderField('Website', 'website', entry.website, 'url')}
                    ${renderField('Subject', 'subject', entry.subject)}
                    ${renderField('Message', 'message', entry.message, 'textarea')}
                    ${renderField('Privacy Policy', 'privacy_status', entry.privacy_status || 'not_configured', 'select')}
                </div>
                
                <div class="cmbsqf-entry-divider"></div>
                
                <div class="cmbsqf-entry-delivery">
                    <label>Email Delivery Status</label>
                    <div>${deliveryStatus}</div>
                </div>
            </div>
            <div class="cmbsqf-modal-footer">
                <div class="cmbsqf-modal-footer-left">
                    <button type="button" class="button cmbsqf-entry-edit-btn">
                        <span class="dashicons dashicons-edit"></span> Edit
                    </button>
                    <button type="button" class="button cmbsqf-entry-resend-btn">
                        <span class="dashicons dashicons-email"></span> Resend Email
                    </button>
                </div>
                <div class="cmbsqf-modal-footer-right">
                    <button type="button" class="button cmbsqf-entry-delete-btn">
                        <span class="dashicons dashicons-trash"></span> Delete
                    </button>
                </div>
            </div>
        </div>
    `;

    document.body.appendChild(overlay);
    document.body.style.overflow = 'hidden';
}

/**
 * Render field with readonly input
 */
function renderField(label, name, value, type = 'text') {
    return `
        <div class="cmbsqf-entry-field" data-field="${name}" data-type="${type}">
            <label for="entry-${name}">${escapeHtml(label)}</label>
            ${renderInput(name, value, type, true)}
        </div>
    `;
}

/**
 * Render input based on type
 */
function renderInput(name, value, type, readonly = false) {
    const readonlyAttr = readonly ? 'readonly' : '';
    const disabledAttr = readonly ? 'disabled' : '';

    if (type === 'textarea') {
        return `<textarea id="entry-${name}" name="${name}" class="cmbsqf-textarea" rows="6" ${readonlyAttr}>${escapeHtml(value || '')}</textarea>`;
    } else if (type === 'select') {
        const options = [
            { value: 'not_configured', label: 'Not configured' },
            { value: 'not_accepted', label: 'Not accepted' },
            { value: 'accepted', label: 'Accepted' }
        ];
        const optionsHTML = options.map(opt => {
            const selected = opt.value === value ? 'selected' : '';
            return `<option value="${opt.value}" ${selected}>${opt.label}</option>`;
        }).join('');
        return `<select id="entry-${name}" name="${name}" class="cmbsqf-select" ${disabledAttr}>${optionsHTML}</select>`;
    } else {
        return `<input type="${type}" id="entry-${name}" name="${name}" class="cmbsqf-input" value="${escapeHtml(value || '')}" ${readonlyAttr}>`;
    }
}

/**
 * Enable edit mode
 */
export function enableEditMode() {
    isEditMode = true;

    document.querySelectorAll('.cmbsqf-entry-field input, .cmbsqf-entry-field textarea').forEach(input => {
        input.removeAttribute('readonly');
    });

    document.querySelectorAll('.cmbsqf-entry-field select').forEach(select => {
        select.removeAttribute('disabled');
    });

    const footer = document.querySelector('.cmbsqf-modal-footer');
    footer.innerHTML = `
        <div class="cmbsqf-modal-footer-left">
            <button type="button" class="button button-primary cmbsqf-entry-save-btn">
                <span class="dashicons dashicons-yes"></span> Save Changes
            </button>
            <button type="button" class="button cmbsqf-entry-cancel-btn">
                <span class="dashicons dashicons-no-alt"></span> Cancel
            </button>
        </div>
        <div class="cmbsqf-modal-footer-right">
            <button type="button" class="button cmbsqf-entry-delete-btn">
                <span class="dashicons dashicons-trash"></span> Delete
            </button>
        </div>
    `;
}

/**
 * Cancel edit mode
 */
export function cancelEdit() {
    isEditMode = false;

    const fields = ['name', 'surname', 'email', 'phone', 'organization', 'position', 'website', 'subject', 'message', 'privacy_status'];
    fields.forEach(fieldName => {
        const field = document.querySelector(`[data-field="${fieldName}"]`);
        if (field) {
            const input = field.querySelector('[name]');
            if (input) {
                if (input.tagName === 'TEXTAREA' || input.tagName === 'INPUT') {
                    input.value = currentEntry[fieldName] || '';
                } else if (input.tagName === 'SELECT') {
                    input.value = currentEntry[fieldName] || 'not_configured';
                }
            }
        }
    });

    document.querySelectorAll('.cmbsqf-entry-field input, .cmbsqf-entry-field textarea').forEach(input => {
        input.setAttribute('readonly', 'readonly');
    });

    document.querySelectorAll('.cmbsqf-entry-field select').forEach(select => {
        select.setAttribute('disabled', 'disabled');
    });

    const footer = document.querySelector('.cmbsqf-modal-footer');
    footer.innerHTML = `
        <div class="cmbsqf-modal-footer-left">
            <button type="button" class="button cmbsqf-entry-edit-btn">
                <span class="dashicons dashicons-edit"></span> Edit
            </button>
            <button type="button" class="button cmbsqf-entry-resend-btn">
                <span class="dashicons dashicons-email"></span> Resend Email
            </button>
        </div>
        <div class="cmbsqf-modal-footer-right">
            <button type="button" class="button cmbsqf-entry-delete-btn">
                <span class="dashicons dashicons-trash"></span> Delete
            </button>
        </div>
    `;
}

/**
 * Save changes
 */
export async function saveChanges() {
    const updateData = {};
    const fields = ['name', 'surname', 'email', 'phone', 'organization', 'position', 'website', 'subject', 'message', 'privacy_status'];

    fields.forEach(fieldName => {
        const field = document.querySelector(`[data-field="${fieldName}"]`);
        if (field) {
            const input = field.querySelector('[name]');
            if (input && input.value !== (currentEntry[fieldName] || '')) {
                updateData[fieldName] = input.value.trim();
            }
        }
    });

    if (Object.keys(updateData).length === 0) {
        showToast('No changes to save', 'info');
        cancelEdit();
        return;
    }

    if (updateData.email && !isValidEmail(updateData.email)) {
        alert('Please enter a valid email address.');
        return;
    }

    try {
        const response = await fetchWithNonce(getRestUrl(`/entries/${currentEntry.id}?form_id=${window.cmbsqfSettings.formId}`), {
            method: 'PATCH',
            body: JSON.stringify(updateData),
        });

        const data = await response.json();

        if (response.ok && data.success) {
            Object.assign(currentEntry, updateData);
            showToast('Changes saved successfully', 'success');
            updateTableRow(currentEntry.id, updateData);
            cancelEdit();
        } else {
            alert('Error: ' + (data.message || 'Failed to save changes'));
        }
    } catch (error) {
        console.error('Error saving changes:', error);
        alert('Error saving changes. Please try again.');
    }
}

/**
 * Resend email notification
 */
export async function resendEmail(entryId = null, email = null, updateUI = true) {
    const id = entryId || (currentEntry ? currentEntry.id : null);

    if (!id) {
        console.error('resendEmail: No entry ID provided');
        return false;
    }

    if (updateUI) {
        const recipientEmail = email || (currentEntry ? currentEntry.email : 'this entry');
        if (!confirm(`Resend email notification to ${recipientEmail}?`)) {
            return false;
        }
    }

    try {
        const response = await fetchWithNonce(getRestUrl(`/entries/${id}/resend?form_id=${window.cmbsqfSettings.formId}`), {
            method: 'POST',
        });

        const data = await response.json();

        if (response.ok && data.success) {
            if (updateUI) {
                showToast('Email sent successfully', 'success');

                const deliveryDiv = document.querySelector('.cmbsqf-entry-delivery > div');
                if (deliveryDiv) {
                    deliveryDiv.innerHTML = '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-sent">●</span> Email sent successfully';
                }
            }

            updateTableNotifiedColumn(id, 'sent');
            return true;
        } else {
            if (updateUI) {
                alert('Error: ' + (data.message || 'Failed to resend email'));
            }
            return false;
        }
    } catch (error) {
        console.error('Error resending email:', error);
        if (updateUI) {
            alert('Error resending email. Please try again.');
        }
        return false;
    }
}

/**
 * Delete entry
 */
export async function deleteEntry() {
    if (!confirm('Are you sure you want to delete this entry? This action cannot be undone.')) {
        return;
    }

    try {
        const response = await fetchWithNonce(getRestUrl(`/entries/${currentEntry.id}?form_id=${window.cmbsqfSettings.formId}`), {
            method: 'DELETE',
        });

        const data = await response.json();

        if (response.ok && data.success) {
            closeModal();

            const row = document.querySelector(`tr[data-entry-id="${currentEntry.id}"]`);
            if (row) {
                row.style.backgroundColor = '#ffcccc';
                setTimeout(() => {
                    row.remove();

                    const tbody = document.querySelector('.wp-list-table tbody');
                    if (tbody && tbody.children.length === 0) {
                        window.location.reload();
                    }
                }, 500);
            }
        } else {
            alert('Error: ' + (data.message || 'Failed to delete entry'));
        }
    } catch (error) {
        console.error('Error deleting entry:', error);
        alert('Error deleting entry. Please try again.');
    }
}

/**
 * Update table row after saving
 */
function updateTableRow(entryId, updateData) {
    const row = document.querySelector(`tr[data-entry-id="${entryId}"]`);
    if (!row) return;

    if (updateData.hasOwnProperty('name') || updateData.hasOwnProperty('surname')) {
        const nameCell = row.querySelector('.column-name');
        if (nameCell) {
            const link = nameCell.querySelector('.cmbsqf-entry-link strong');
            if (link) {
                const name = currentEntry.name || '';
                const surname = currentEntry.surname || '';
                const fullName = (name + ' ' + surname).trim() || '—';
                link.textContent = fullName;
            }
        }
    }

    if (updateData.hasOwnProperty('email')) {
        const emailCell = row.querySelector('.column-email');
        if (emailCell) {
            emailCell.textContent = currentEntry.email || '—';
        }
    }

    if (updateData.hasOwnProperty('privacy_status')) {
        const privacyCell = row.querySelector('.column-privacy');
        if (privacyCell) {
            const statusHTML = getPrivacyStatusHTML(currentEntry.privacy_status);
            privacyCell.innerHTML = statusHTML;
        }
    }
}

/**
 * Update notified column in table
 */
function updateTableNotifiedColumn(entryId, status) {
    const row = document.querySelector(`tr[data-entry-id="${entryId}"]`);
    if (!row) return;

    const notifiedCell = row.querySelector('.column-notified');
    if (!notifiedCell) return;

    let bulletHTML = '';
    switch (status) {
        case 'sent':
            bulletHTML = '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-sent" title="Email sent successfully">●</span>';
            break;
        case 'failed':
            bulletHTML = '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-failed" title="Failed to send">●</span>';
            break;
        case 'not_configured':
        default:
            bulletHTML = '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-na" title="Email notification not configured">●</span>';
            break;
    }

    notifiedCell.innerHTML = bulletHTML;
}

/**
 * Get delivery status HTML
 */
function getDeliveryStatusHTML(entry) {
    const status = entry.email_delivery || 'not_configured';

    switch (status) {
        case 'sent':
            return '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-sent">●</span> Email sent successfully';
        case 'failed':
            const error = entry.email_delivery_error || 'Unknown error';
            return `<span class="cmbsqf-delivery-bullet cmbsqf-delivery-failed">●</span> Failed: ${escapeHtml(error)}`;
        case 'not_configured':
        default:
            return '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-na">●</span> Email notification not configured';
    }
}

/**
 * Get privacy status HTML for table
 */
function getPrivacyStatusHTML(status) {
    switch (status) {
        case 'accepted':
            return '<span style="color: #00a32a;">✓ Accepted</span>';
        case 'not_accepted':
            return '<span style="color: #d63638;">✗ Not accepted</span>';
        case 'not_configured':
        default:
            return '<span style="color: #646970;">— Not configured</span>';
    }
}
