/**
 * Entries Bulk Actions
 * Handle bulk delete and bulk resend operations
 */

import { fetchWithNonce, getRestUrl, getEmailFromRow, escapeHtml, showToast } from './entries-utils.js';

/**
 * Handle bulk action based on selected action type
 */
export async function handleBulkAction(e, position) {
    const selectorId = position === 'top' ? 'bulk-action-selector-top' : 'bulk-action-selector-bottom';
    const selector = document.getElementById(selectorId);

    if (!selector || (selector.value !== 'delete' && selector.value !== 'resend_user_email')) {
        return; // Let default handler proceed
    }

    e.preventDefault();

    const checkboxes = document.querySelectorAll('input[name="entry_ids[]"]:checked');
    if (checkboxes.length === 0) {
        return;
    }

    const ids = Array.from(checkboxes).map(cb => parseInt(cb.value));

    if (selector.value === 'delete') {
        await handleBulkDelete(ids);
    } else if (selector.value === 'resend_user_email') {
        await handleBulkResend(ids, position);
    }
}

/**
 * Handle bulk delete operation
 */
async function handleBulkDelete(ids) {
    if (!confirm('Are you sure you want to delete the selected entries? This action cannot be undone.')) {
        return;
    }

    let successCount = 0;
    for (const id of ids) {
        try {
            const response = await fetchWithNonce(getRestUrl(`/entries/${id}?form_id=${window.cmbsqfSettings.formId}`), {
                method: 'DELETE',
            });

            const data = await response.json();

            if (response.ok && data.success) {
                successCount++;
                const row = document.querySelector(`tr[data-entry-id="${id}"]`);
                if (row) {
                    row.style.backgroundColor = '#ffcccc';
                    row.style.transition = 'background-color 0.5s';
                    setTimeout(() => row.remove(), 500);
                }
            } else {
                console.error('Failed to delete entry:', id, data.message);
            }
        } catch (error) {
            console.error('Error deleting entry:', id, error);
        }
    }

    setTimeout(() => {
        const tbody = document.querySelector('.wp-list-table tbody');
        if (tbody && tbody.children.length === 0) {
            window.location.reload();
        }
    }, 600);
}

/**
 * Handle bulk resend email operation
 */
async function handleBulkResend(ids, position) {
    if (!confirm(`Resend email notification to ${ids.length} selected entries?`)) {
        return;
    }

    let successCount = 0;
    let failCount = 0;
    const successList = [];
    const failureList = [];

    const applyBtn = position === 'top' ? document.getElementById('doaction') : document.getElementById('doaction2');
    const originalText = applyBtn.value;
    applyBtn.value = 'Processing...';
    applyBtn.disabled = true;

    for (const id of ids) {
        try {
            const response = await fetchWithNonce(getRestUrl(`/entries/${id}/resend`), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    form_id: window.cmbsqfSettings.formId
                }),
            });

            const data = await response.json();
            const email = getEmailFromRow(id);

            if (response.ok && data.success) {
                successCount++;
                successList.push({ id, email });
                updateTableNotifiedColumn(id, 'sent');
            } else {
                failCount++;
                const errorMsg = data.message || 'Unknown error';
                failureList.push({ id, email, error: errorMsg });
                updateTableNotifiedColumn(id, 'failed');
            }
        } catch (error) {
            failCount++;
            const email = getEmailFromRow(id);
            failureList.push({ id, email, error: error.message });
            updateTableNotifiedColumn(id, 'failed');
        }
    }

    applyBtn.value = originalText;
    applyBtn.disabled = false;

    showBulkResultsModal({
        successCount,
        failCount,
        successList,
        failureList,
        total: ids.length
    });
}

/**
 * Update notified column in table for a specific entry
 */
function updateTableNotifiedColumn(entryId, status) {
    const row = document.querySelector(`tr[data-entry-id="${entryId}"]`);
    if (!row) return;

    const notifiedCell = row.querySelector('.column-notified');
    if (!notifiedCell) return;

    let bulletHTML = '';
    switch (status) {
        case 'sent':
            bulletHTML = '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-sent" title="Email sent successfully">●</span>';
            break;
        case 'failed':
            bulletHTML = '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-failed" title="Failed to send">●</span>';
            break;
        case 'not_configured':
        default:
            bulletHTML = '<span class="cmbsqf-delivery-bullet cmbsqf-delivery-na" title="Email notification not configured">●</span>';
            break;
    }

    notifiedCell.innerHTML = bulletHTML;
}

/**
 * Show bulk action results in a professional modal
 */
function showBulkResultsModal(results) {
    const overlay = document.createElement('div');
    overlay.className = 'cmbsqf-modal-overlay cmbsqf-results-modal';

    const successHTML = results.successList.map(item =>
        `<div class="result-item success"><span class="dashicons dashicons-yes-alt"></span> Entry #${item.id} - ${escapeHtml(item.email)}</div>`
    ).join('');

    const failuresHTML = results.failureList.map(item =>
        `<div class="result-item failure">
            <span class="dashicons dashicons-warning"></span>
            <div>
                <strong>Entry #${item.id} - ${escapeHtml(item.email)}</strong>
                <p class="error-message">${escapeHtml(item.error)}</p>
            </div>
        </div>`
    ).join('');

    overlay.innerHTML = `
        <div class="cmbsqf-modal-container cmbsqf-results-container">
            <div class="cmbsqf-modal-header">
                <h2><span class="dashicons dashicons-email-alt"></span> Bulk Email Resend Results</h2>
                <button type="button" class="cmbsqf-modal-close" aria-label="Close">
                    <span class="dashicons dashicons-no-alt"></span>
                </button>
            </div>
            <div class="cmbsqf-modal-body cmbsqf-results-body">
                <div class="results-summary">
                    <div class="summary-box success-box">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <div>
                            <strong>${results.successCount}</strong>
                            <span>Successful</span>
                        </div>
                    </div>
                    <div class="summary-box failure-box">
                        <span class="dashicons dashicons-dismiss"></span>
                        <div>
                            <strong>${results.failCount}</strong>
                            <span>Failed</span>
                        </div>
                    </div>
                    <div class="summary-box total-box">
                        <span class="dashicons dashicons-email"></span>
                        <div>
                            <strong>${results.total}</strong>
                            <span>Total</span>
                        </div>
                    </div>
                </div>

                ${results.successCount > 0 ? `
                    <div class="results-section">
                        <h3><span class="dashicons dashicons-yes-alt"></span> Successful Entries</h3>
                        <div class="results-list">${successHTML}</div>
                    </div>
                ` : ''}

               ${results.failCount > 0 ? `
                    <div class="results-section">
                        <h3><span class="dashicons dashicons-warning"></span> Failed Entries</h3>
                        <div class="results-list">${failuresHTML}</div>
                    </div>
                ` : ''}
            </div>
            <div class="cmbsqf-modal-footer">
                <button type="button" class="button cmbsqf-copy-report-btn">
                    <span class="dashicons dashicons-clipboard"></span> Copy Report
                </button>
                <button type="button" class="button button-primary cmbsqf-close-results-btn">Close</button>
            </div>
        </div>
    `;

    overlay.querySelector('.cmbsqf-modal-close').addEventListener('click', () => overlay.remove());
    overlay.querySelector('.cmbsqf-close-results-btn').addEventListener('click', () => overlay.remove());
    overlay.querySelector('.cmbsqf-copy-report-btn').addEventListener('click', () => {
        const report = generateTextReport(results);
        navigator.clipboard.writeText(report).then(() => {
            showToast('Report copied to clipboard', 'success');
        });
    });

    document.body.appendChild(overlay);
}

/**
 * Generate text report for copying
 */
function generateTextReport(results) {
    let report = `BULK EMAIL RESEND REPORT\n`;
    report += `========================\n\n`;
    report += `Total Processed: ${results.total}\n`;
    report += `Successful: ${results.successCount}\n`;
    report += `Failed: ${results.failCount}\n\n`;

    if (results.successList.length > 0) {
        report += `SUCCESSFUL ENTRIES:\n`;
        results.successList.forEach(item => {
            report += `  ✓ Entry #${item.id} - ${item.email}\n`;
        });
        report += `\n`;
    }

    if (results.failureList.length > 0) {
        report += `FAILED ENTRIES:\n`;
        results.failureList.forEach(item => {
            report += `  ✗ Entry #${item.id} - ${item.email}\n`;
            report += `    Error: ${item.error}\n`;
        });
    }

    return report;
}
