/**
 * Entries Controller Module (Entry Point)
 * Coordinates all entry management functionality
 */

import { handleBulkAction } from './entries/entries-bulk.js';
import {
    openModal,
    closeModal,
    enableEditMode,
    cancelEdit,
    saveChanges,
    resendEmail,
    deleteEntry
} from './entries/entries-modal.js';
import { downloadDatabase, clearAllEntries, showToast } from './entries/entries-utils.js';

/**
 * Initialize entries manager
 */
function init() {
    bindEvents();
}

/**
 * Bind event listeners
 */
function bindEvents() {
    // Click on Name link to open modal
    document.addEventListener('click', function (e) {
        const link = e.target.closest('.cmbsqf-entry-link');
        if (link) {
            e.preventDefault();
            const entryId = parseInt(link.getAttribute('data-entry-id'));
            openModal(entryId);
        }
    });

    // Modal close events
    document.addEventListener('click', function (e) {
        // Close button
        if (e.target.closest('.cmbsqf-modal-close')) {
            closeModal();
        }

        // Overlay click
        if (e.target.classList.contains('cmbsqf-modal-overlay')) {
            closeModal();
        }

        // Edit button
        if (e.target.closest('.cmbsqf-entry-edit-btn')) {
            e.preventDefault();
            enableEditMode();
        }

        // Save Changes button
        if (e.target.closest('.cmbsqf-entry-save-btn')) {
            e.preventDefault();
            saveChanges();
        }

        // Cancel button
        if (e.target.closest('.cmbsqf-entry-cancel-btn')) {
            e.preventDefault();
            cancelEdit();
        }

        // Resend Email button
        if (e.target.closest('.cmbsqf-entry-resend-btn')) {
            e.preventDefault();
            resendEmail();
        }

        // Delete Entry button
        if (e.target.closest('.cmbsqf-entry-delete-btn')) {
            e.preventDefault();
            deleteEntry();
        }
    });

    // ESC key to close modal
    document.addEventListener('keydown', function (e) {
        if (e.key === 'Escape' && document.querySelector('.cmbsqf-modal-overlay')) {
            closeModal();
        }
    });

    // Bulk Actions (Top)
    const doActionTop = document.getElementById('doaction');
    if (doActionTop) {
        doActionTop.addEventListener('click', function (e) {
            handleBulkAction(e, 'top');
        });
    }

    // Bulk Actions (Bottom)
    const doActionBottom = document.getElementById('doaction2');
    if (doActionBottom) {
        doActionBottom.addEventListener('click', function (e) {
            handleBulkAction(e, 'bottom');
        });
    }

    // Download Database button
    const downloadDbBtn = document.querySelector('.cmbsqf-download-database');
    if (downloadDbBtn) {
        downloadDbBtn.addEventListener('click', function (e) {
            e.preventDefault();
            const formId = this.getAttribute('data-form-id');
            if (formId) {
                downloadDatabase(parseInt(formId));
            }
        });
    }

    // Clear All Entries button
    const clearAllBtn = document.querySelector('.cmbsqf-clear-all-entries');
    if (clearAllBtn) {
        clearAllBtn.addEventListener('click', function (e) {
            e.preventDefault();
            const formId = this.getAttribute('data-form-id');
            const formName = this.getAttribute('data-form-name');
            if (formId && formName) {
                clearAllEntries(parseInt(formId), formName);
            }
        });
    }

    // CSV Export Modal
    const csvExportBtn = document.querySelector('.cmbsqf-export-csv');
    const csvModal = document.getElementById('cmbsqf-csv-modal');
    const csvModalClose = document.querySelector('.cmbsqf-csv-modal-close');

    if (csvExportBtn && csvModal) {
        // Open modal
        csvExportBtn.addEventListener('click', function (e) {
            e.preventDefault();
            csvModal.style.display = 'flex';
            document.body.style.overflow = 'hidden';
        });

        // Close modal function
        const closeCsvModal = () => {
            csvModal.style.display = 'none';
            document.body.style.overflow = '';
        };

        // Close button
        if (csvModalClose) {
            csvModalClose.addEventListener('click', closeCsvModal);
        }

        // Close on overlay click
        csvModal.addEventListener('click', function (e) {
            if (e.target === csvModal) {
                closeCsvModal();
            }
        });

        // Click on option labels to export directly
        const csvOptions = csvModal.querySelectorAll('.cmbsqf-csv-option label');
        csvOptions.forEach(label => {
            label.addEventListener('click', function (e) {
                e.preventDefault();

                const formId = csvExportBtn.getAttribute('data-form-id');

                if (!formId) {
                    console.error('Form ID not found');
                    return;
                }

                // Get selected delimiter from radio button
                const radioInput = this.querySelector('input[name="csv_delimiter"]');
                const delimiter = radioInput ? radioInput.value : 'comma';

                try {
                    // Construct admin-post.php URL with nonce
                    const url = new URL(window.location.origin + '/wp-admin/admin-post.php');
                    url.searchParams.set('action', 'cmbsqf_download_csv');
                    url.searchParams.set('form_id', formId);
                    url.searchParams.set('delimiter', delimiter);
                    url.searchParams.set('_wpnonce', window.cmbsqfSettings.downloadCsvNonce);

                    // Trigger download via navigation
                    window.location.href = url.toString();

                    closeCsvModal();
                } catch (error) {
                    console.error('CSV export error:', error);
                    showToast('Failed to export CSV', 'error');
                }
            });
        });
    }
}

// Initialize on DOM ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
} else {
    init();
}
