<?php
/**
 * Download Handler
 *
 * Purpose: Handle file downloads (DB, CSV) via admin_post_ actions
 * Location: /admin-post/class-download-handler.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class CMBSQF_Download_Handler {

    /**
     * Register admin_post hooks.
     */
    public static function register() {
        add_action('admin_post_cmbsqf_download_db', [__CLASS__, 'handle_db_download']);
        add_action('admin_post_cmbsqf_download_csv', [__CLASS__, 'handle_csv_download']);
    }

    /**
     * Handle database download.
     */
    public static function handle_db_download() {
        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'cmbsqf_download_db')) {
            wp_die(esc_html__('Security check failed.', 'cmb-sqlite-form'), 403);
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to download this file.', 'cmb-sqlite-form'), 403);
        }

        // Get form ID
        $form_id = isset($_GET['form_id']) ? intval($_GET['form_id']) : 0;

        if ($form_id <= 0) {
            wp_die(esc_html__('Invalid form ID.', 'cmb-sqlite-form'), 400);
        }

        // Get database file
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-core.php';
        $db_path = CMBSQF_DB_Core::get_form_entries_db_path($form_id);

        if (!file_exists($db_path)) {
            wp_die(esc_html__('Database file not found.', 'cmb-sqlite-form'), 404);
        }

        // Serve file
        $filename = 'form-' . $form_id . '-' . gmdate('Y-m-d') . '.db';
        
        header('Content-Type: application/x-sqlite3');
        header('Content-Disposition: attachment; filename="' . rawurlencode($filename) . '"');
        header('Content-Length: ' . filesize($db_path));
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile -- Direct readfile for binary DB download
        readfile($db_path);
        exit;
    }

    /**
     * Handle CSV download.
     */
    public static function handle_csv_download() {
        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'cmbsqf_download_csv')) {
            wp_die(esc_html__('Security check failed.', 'cmb-sqlite-form'), 403);
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to download this file.', 'cmb-sqlite-form'), 403);
        }

        // Get form ID
        $form_id = isset($_GET['form_id']) ? intval($_GET['form_id']) : 0;

        if ($form_id <= 0) {
            wp_die(esc_html__('Invalid form ID.', 'cmb-sqlite-form'), 400);
        }

        // Get delimiter
        $delimiter = isset($_GET['delimiter']) && $_GET['delimiter'] === 'semicolon' ? ';' : ',';

        // Load dependencies
        require_once CMBSQF_PLUGIN_DIR . 'includes/db/class-db-submissions.php';

        // Get submissions
        $submissions = CMBSQF_DB_Submissions::get_all($form_id);

        if (empty($submissions)) {
            wp_die(esc_html__('No entries found to export.', 'cmb-sqlite-form'), 404);
        }

        // Define column order
        $columns = [
            'id',
            'name',
            'surname',
            'email',
            'phone',
            'organization',
            'position',
            'website',
            'subject',
            'message',
            'privacy_status',
            'ip_address',
            'user_agent',
            'created_at',
            'email_delivery',
            'email_delivery_error'
        ];

        // Prepare CSV data
        $csv_data = [];
        
        // Header row with friendly names
        $headers = [
            'ID',
            'Name',
            'Surname',
            'Email',
            'Phone',
            'Organization',
            'Position',
            'Website',
            'Subject',
            'Message',
            'Privacy Status',
            'IP Address',
            'User Agent',
            'Created At',
            'Email Delivery',
            'Email Delivery Error'
        ];
        $csv_data[] = $headers;

        // Data rows
        foreach ($submissions as $entry) {
            $row = [];
            foreach ($columns as $column) {
                $row[] = $entry[$column] ?? '';
            }
            $csv_data[] = $row;
        }

        // Generate CSV content with proper Windows line endings
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Using php://temp for in-memory CSV generation
        $output = fopen('php://temp', 'r+');
        
        // Write BOM for Excel UTF-8 compatibility
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite -- Writing to in-memory stream
        fwrite($output, "\xEF\xBB\xBF");
        
        foreach ($csv_data as $row) {
            // Manually build CSV line to ensure proper line endings and escaping
            $line = [];
            foreach ($row as $field) {
                // Escape fields that contain delimiter, quotes, or line breaks
                if (strpos($field, $delimiter) !== false || strpos($field, '"') !== false || strpos($field, "\n") !== false || strpos($field, "\r") !== false) {
                    $field = '"' . str_replace('"', '""', $field) . '"';
                }
                $line[] = $field;
            }
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite -- Writing to in-memory stream
            fwrite($output, implode($delimiter, $line) . "\r\n");
        }
        
        rewind($output);
        $csv_content = stream_get_contents($output);
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing in-memory stream
        fclose($output);

        // Prepare filename
        $filename = 'form-' . $form_id . '-entries-' . gmdate('Y-m-d') . '.csv';

        // Send headers
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . rawurlencode($filename) . '"');
        header('Content-Length: ' . strlen($csv_content));
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Output CSV content and exit
        echo $csv_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV binary output
        exit;
    }
}
